(* ::Package:: *)

(* :Author: Bernd Guenther *)
(* :Package Version: 1.0 *)
(* :Mathematica Version: 12.1.1 *)

   
BeginPackage["LieAlg4BG`"];


ElemMatrices::usage="ElemMatrices[n] lists the elementary matrices (standard basis) of dimension nxn."


LieAlgGL::usage="LieAlgGL[n] returns the structure constants of the general linear algebra in n dimensions with respect to the standard basis."


LieAlgSL::usage="LieAlgSL[n] returns the structure constants of the special linear algebra in n dimensions with respect to the standard basis."


LieAlgSO::usage="LieAlgSO[n] returns the structure constants of the special orthogonal algebra with respect to the standard basis."


LieAlgSU::usage="LieAlgSU[n] returns the structure constants of the special unitary algebra in n dimensions with respect to the standard basis."


LieAlgSp::usage="LieAlgSp[n] returns the structure constants of the symplectic algebra in 2n dimensions with respect to the standard basis."


LieAlgSOpq::usage="LieAlgSOpq[p,q] returns the structure constants of the mixed orthogonal algebra in p+q dimensions with respect to the standard base matrices."


LieAlgTri::usage="LieAlgTri[n] returns the structure constants of the triangular algebra in n dimensions with respect to the standard base matrices."


LieAlgSuTri::usage="LieAlgSuTri[n] returns the structure constants of the strictly upper triangular algebra in n dimensions with respect to the standard base matrices."


LieAlgExcG2::usage="LieAlgExcG2 returns the structure constants of the exceptional Lie algebra G2."


LieAlgExcE::usage="LieAlgExcE[n] returns the structure constants of the exceptional Lie algebras En for n=6,7,8."


LieAlgExcF4::usage="LieAlgExcF4 returns the structure constants of the exceptional Lie algebra of type F4."


LieMatricesSL::usage="LieMatricesSL[n] returns the standard base matrices of the special linear algebra in n dimensions."


LieMatricesSO::usage="LieMatricesSO[n] returns the standard base matrices of the orthogonal algebra (the antisymmetric matrices) in n dimensions."


LieMatricesSU::usage="LieMatricesSU[n] returns the standard base matrices of the special unitary algebra in n dimensions."


LieMatricesSp::usage="LieMatricesSp[n] returns the standard base matrices of the symplectic algebra in 2n dimensions."


LieMatricesSOpq::usage="LieMatricesSOpq[p,q] returns standard base matrices of the mixed orthogonal algebra in p+q dimensions."


LieMatricesTri::usage="LieMatricesTri[n] returns the standard base matrices of the triangular Lie algebra in n dimensions."


LieMatricesExcG2::usage="LieMatricesExcG2 returns the base matrices of the exceptional Lie algebra G2."


LieAxiomCheck::usage="LieAxiomCheck[x] is true if and only if x is a three dimensional tensor satisfying the axioms of Lie structure constants (antisymmetry and Jacobi identity)."


LieCenter::usage="LieCenter[x] computes the center of the Lie algebra with structure constants x as collection of base vectors."


LieKillingForm::usage="LieKillingForm[x] generates the Killing form of the Lie algebra with structure constants x."


LieIsNilpotent::usage="Evaluates if the Lie algebra with structure constants x is nilpotent."


LieIsCartanAlgebra::usage="LieIsCartanAlgebra[liealg,subsp] evaluates if the subspace spanned by the rowvectors subsp is a Cartan subalgebra of liealg."


LieFindCartanSubalgebra::usage="LieFindCartanSubalgebra[liealg] returns a base (set of row vectors) for a Cartan subalgebra of the Lie algebra with structure constants liealg."


JordanBlockDecompose::usage="JordanBlockDecompose[x] returns a list of triples {i,j,p} (one for each primary block of a square matrix x), where i is the inclusion map of the component space, p is the projection map of total space onto the component space and j is the Jordan normal matrix j=p.x.i."


LieReprRootDecomposition::usage="For a Lie algebra x with Cartan subalgebra csa, and for a matrix representation repr of x, LieReprRootDecomposition[liealg,csa,repr] returns the root space decomposition of x als list of triples {i,v,p}, where v is a weight vector, i is the inclusion operator of the corresponding root space and p is the projection operator onto the corresponding root space."


LieRootDecomposition::usage="For a Lie algebra x, LieRootDecomposition[x] returns the root space decomposition of x als list of triples {i,v,p}, where v is a root vector (0 included), i is the inclusion operator of the corresponding root space and p is the projection operator onto the corresponding root space."


LieChevalleyBase::usage="For a semisimple Lie algebra x, LieChevalleyBase[x] returns a six tuple {norms,cartanMatrix,dynkinGraph,cartanAlgebra,chevalleyRootSystem,structConst}, where norms is the list of the norm squares of the root base vectors, chevalleyRootSystem is a list of triples {\[Alpha],x,y} consisting of a positive root form \[Alpha], a positive eigenvector x and a negative eigenvector y, and structConst are the structure constants with respect to the Chevalley base consisting of the given base vectors of the Cartan algebra and the root eigenvectors. We order the base vectors starting with the Cartan algebra and then, in pairs, each positive root eigenvector immediately followed by its negative counterpart. Notice that the Cartan algebra and the root forms constitute a dual pair of vector spaces, the corresponding evaluation bilinear form with respect to the given basis being the Cartan matrix.
LieChevalleyBase[x,csa] returns data as above, but for a provided Cartan subalgebra csa (which may undergo a base transformation but is preseved as a whole.)"


LieCartanMtrx2RootSystem::usage="LieCartanMtrx2RootSystem[ctrMtrx] returns the list of positive roots generated by the given Cartan matrix."


LieRepresentationCheck::usage="LieRepresentationCheck[liealg,repr] returns True if the matrices repr constitute a representations of the Lie algebra with structure constants liealg, as images of the base vectors."


Begin["`Private`"]


sparseIdentityMatrix[n_] := SparseArray[{{i_, i_}->1}, {n,n}];


ElemMatrices[n_Integer]:=Module[{i,j},SparseArray[Flatten[Table[{{n*(i-1)+j,i,j}->1},{i,n},{j,n}]],{n^2,n,n}]]/;n>=1


LieAlgGL[n_Integer]:=Module[{a,b,d},SparseArray[Flatten[Apply[{Table[If[#2!=#3,{#1,#2,#3}->#4,Nothing],{a,n},{b,n},{d,n}],Table[If[#2!=#3,{#1,#3,#2}->-#4,Nothing],{a,n},{b,n},{d,n}]}&,{{n(a-1)+d,n (a-1)+b,n (b-1)+d,1}},{1}]],{n^2,n^2,n^2}]]/;n>=1


LieAlgSL[n_Integer]:=Module[{a,b,c},SparseArray[Join[Flatten[Apply[{Table[If[#4!=0,{#1,#2,#3}->#4,Nothing],{a,n},{b,n},{c,n}],Table[If[#4!=0,{#1,#3,#2}->-#4,Nothing],{a,n},{b,n},{c,n}]}&,{{n(a-1)+c,n (a-1)+b,n (b-1)+c,If[a!=b&&a!=c&&b!=c,1,0]},{n(c-1)+c,n(a-1)+b,n(b-1)+a,If[a<=c&&c<b,1,0]}},{1}]],Flatten[Apply[{Table[If[#4!=0,{#1,#2,#3}->#4,Nothing],{b,n},{a,n-1}],Table[If[#4!=0,{#1,#3,#2}->-#4,Nothing],{b,n},{a,n-1}]}&,{{n(a-1)+b,n(a-1)+a,n(a-1)+b,If[b!=a&&b!=a+1,1,0]},{n(b-1)+a,n(a-1)+a,n(b-1)+a,If[b!=a&&b!=a+1,-1,0]},{n a+b,n(a-1)+a,n a+b,If[b!=a&&b!=a+1,-1,0]},{n (b-1)+a+1,n(a-1)+a,n (b-1)+a+1,If[b!=a&&b!=a+1,1,0]}},{1}]],Flatten[Apply[{Table[If[#4!=0,{#1,#2,#3}->#4,Nothing],{a,n-1}],Table[If[#4!=0,{#1,#3,#2}->-#4,Nothing],{a,n-1}]}&,{{n(a-1)+a+1,n(a-1)+a,n(a-1)+a+1,2},{n a+a,n(a-1)+a,n a+a,-2}},{1}]]],{n^2-1,n^2-1,n^2-1}]]/;n>=2


LieAlgSO[n_Integer]:=Module[{a,b,c},SparseArray[Flatten[Apply[{Table[{#1,#2,#3}->#4,{c,n},{b,c-1},{a,b-1}],Table[{#1,#3,#2}->-#4,{c,n},{b,c-1},{a,b-1}]}&,{{(c-1)(c-2)/2+b,(b-1)(b-2)/2+a,(c-1)(c-2)/2+a,1},{(c-1)(c-2)/2+a,(b-1)(b-2)/2+a,(c-1)(c-2)/2+b,-1},{(b-1)(b-2)/2+a,(c-1)(c-2)/2+a,(c-1)(c-2)/2+b,1}},{1}]],{n (n-1)/2,n (n-1)/2,n (n-1)/2}]]/;n>=2


LieAlgSU[n_Integer]:=Module[{a,b,c},SparseArray[Join[Flatten[Apply[{Table[{#1,#2,#3}->#4,{c,n},{b,c-1},{a,b-1}],Table[{#1,#3,#2}->-#4,{c,n},{b,c-1},{a,b-1}]}&,{{(c-1)^2+2b,(b-1)^2+2a,(c-1)^2+2a,1/2},{(c-1)^2+2b,(b-1)^2+2a-1,(c-1)^2+2a-1,1/2},{(c-1)^2+2b-1,(b-1)^2+2a,(c-1)^2+2a-1,1/2},{(c-1)^2+2b-1,(b-1)^2+2a-1,(c-1)^2+2a,-1/2},{(c-1)^2+2a,(b-1)^2+2a,(c-1)^2+2b,-1/2},{(c-1)^2+2a-1,(b-1)^2+2a-1,(c-1)^2+2b,-1/2},{(c-1)^2+2a-1,(b-1)^2+2a,(c-1)^2+2b-1,-1/2},{(c-1)^2+2a,(b-1)^2+2a-1,(c-1)^2+2b-1,1/2},{(b-1)^2+2a,(c-1)^2+2a,(c-1)^2+2b,1/2},
{(b-1)^2+2a-1,(c-1)^2+2a-1,(c-1)^2+2b,1/2},{(b-1)^2+2a-1,(c-1)^2+2a,(c-1)^2+2b-1,-1/2},{(b-1)^2+2a,(c-1)^2+2a-1,(c-1)^2+2b-1,1/2},
{(b-1)^2,(c-1)^2+2a,(c-1)^2+2a-1,-1/Sqrt[2b(b-1)]},{(c-1)^2+2a-1,(b-1)^2,(c-1)^2+2a,-1/Sqrt[2b(b-1)]},
{(c-1)^2+2a,(b-1)^2,(c-1)^2+2a-1,1/Sqrt[2b(b-1)]}},{1}]],Flatten[Apply[{Table[If[#4!=0,{#1,#2,#3}->#4,Nothing],{b,n},{a,1,b-1}],Table[If[#4!=0,{#1,#3,#2}->-#4,Nothing],{b,n},{a,1,b-1}]}&,{{(a-1)^2,(b-1)^2+2a,(b-1)^2+2a-1,Sqrt[(a-1)/(2a)]},{(b-1)^2,(b-1)^2+2a,(b-1)^2+2a-1,-Sqrt[b/(2(b-1))]},{(b-1)^2+2 a-1,(b-1)^2,(b-1)^2+2 a,-b/Sqrt[2 b (b-1)]},{(b-1)^2+2 a,(b-1)^2,(b-1)^2+2 a-1,b/Sqrt[2 b (b-1)]},
{(b-1)^2+2a-1,(a-1)^2,(b-1)^2+2a,Sqrt[(a-1)/(2a)]},
{(b-1)^2+2a,(a-1)^2,(b-1)^2+2a-1,-Sqrt[(a-1)/(2a)]}},{1}]]],{n^2-1,n^2-1,n^2-1}]]/;n>=2


LieAlgSp[n_Integer]:=Module[{a,b,c,d},SparseArray[Flatten[Join[Apply[{Table[If[#2!=#3,{#1,#2,#3}->#4,Nothing],{a,n},{b,n},{d,n}],Table[If[#2!=#3,{#1,#3,#2}->-#4,Nothing],{a,n},{b,n},{d,n}]}&,{{n(a-1)+d,n (a-1)+b,j=n (b-1)+d,1}},{1}],Apply[{Table[{#1,#2,#3}->#4,{a,n},{d,n},{b,d-1}],Table[{#1,#3,#2}->-#4,{a,n},{d,n},{b,d-1}]}&,{{n^2+Max[a,d](Max[a,d]-1)/2+Min[a,d],n (a-1)+b,n^2+d (d-1)/2+b,If[a==d,2,1]},{n^2+Max[a,b](Max[a,b]-1)/2+Min[a,b],n (a-1)+d,n^2+d (d-1)/2+b,If[a==b,2,1]},
{n (3n+1)/2+Max[a,d](Max[a,d]-1)/2+Min[a,d],n (b-1)+a,n (3n+1)/2+d (d-1)/2+b,If[a==d,-2,-1]},
{n (3n+1)/2+Max[a,b](Max[a,b]-1)/2+Min[a,b],n (d-1)+a,n (3n+1)/2+d (d-1)/2+b,If[a==b,-2,-1]}},{1}],Apply[{Table[{#1,#2,#3}->#4,{a,n},{b,n}],Table[{#1,#3,#2}->-#4,{a,n},{b,n}]}&,{{n^2+Max[a,b](Max[a,b]-1)/2+Min[a,b],n(a-1)+b,n^2+b (b+1)/2,If[a!=b,1,2]},
{n (3n+1)/2+Max[a,b](Max[a,b]-1)/2+Min[a,b],n(b-1)+a,n (3n+1)/2+b (b+1)/2,If[a!=b,-1,-2]}},{1}],
Apply[{Table[{#1,#2,#3}->#4,{a,n},{b,a,n},{c,a,n}],Table[{#1,#3,#2}->-#4,{a,n},{b,a,n},{c,a,n}]}&,{{n(b-1)+c,n^2+b (b-1)/2+a,n (3n+1)/2+c (c-1)/2+a,1}},{1}],Apply[{Table[{#1,#2,#3}->#4,{c,n},{b,c},{a,b}],Table[{#1,#3,#2}->-#4,{c,n},{b,c},{a,b}]}&,{{n(c-1)+a,n^2+c (c-1)/2+b,n (3n+1)/2+b (b-1)/2+a,1},{n(a-1)+c,n^2+b (b-1)/2+a,n (3n+1)/2+c (c-1)/2+b,1}},{1}],Apply[{Table[{#1,#2,#3}->#4,{c,n},{a,c},{b,c}],Table[{#1,#3,#2}->-#4,{c,n},{a,c},{b,c}]}&,{{n(a-1)+b,n^2+c (c-1)/2+a,n (3n+1)/2+c (c-1)/2+b,1}},{1}]]],{n(2n+1),n(2n+1),n(2n+1)}]]/;n>=1


LieAlgTri[n_Integer]:=Module[{i,j,k},SparseArray[Flatten[Apply[{Table[If[#4!=0,{#1,#2,#3}->#4,Nothing],{k,n},{j,k},{i,j}],Table[If[#4!=0,{#1,#3,#2}->-#4,Nothing],{k,n},{j,k},{i,j}]}&,{{k (k-1)/2+i,j (j-1)/2+i,k (k-1)/2+j,If[i==k,0,1]}},{1}]],{n (n+1)/2,n (n+1)/2,n (n+1)/2}]]/;n>=1


LieAlgSOpq[p_Integer,q_Integer]:=Module[{a,b,c},SparseArray[Flatten[Join[Apply[{Table[{#1,#2,#3}->#4,{c,p},{b,c-1},{a,b-1}],Table[{#1,#3,#2}->-#4,{c,p},{b,c-1},{a,b-1}]}&,{{(c-1)(c-2)/2+b,(b-1)(b-2)/2+a,(c-1)(c-2)/2+a,1},{(c-1)(c-2)/2+a,(b-1)(b-2)/2+a,(c-1)(c-2)/2+b,-1},{(b-1)(b-2)/2+a,(c-1)(c-2)/2+a,(c-1)(c-2)/2+b,1}},{1}],Apply[{Table[{#1,#2,#3}->#4,{c,q},{b,c-1},{a,b-1}],Table[{#1,#3,#2}->-#4,{c,q},{b,c-1},{a,b-1}]}&,{{p (p-1)/2+(c-1)(c-2)/2+b,p (p-1)/2+(b-1)(b-2)/2+a,p (p-1)/2+(c-1)(c-2)/2+a,1},{p (p-1)/2+(c-1)(c-2)/2+a,p (p-1)/2+(b-1)(b-2)/2+a,p (p-1)/2+(c-1)(c-2)/2+b,-1},{p (p-1)/2+(b-1)(b-2)/2+a,p (p-1)/2+(c-1)(c-2)/2+a,p (p-1)/2+(c-1)(c-2)/2+b,1}},{1}],Apply[{Table[{#1,#2,#3}->#4,{c,q},{b,p},{a,b-1}],Table[{#1,#3,#2}->-#4,{c,q},{b,p},{a,b-1}]}&,{{p (p-1)/2+q (q-1)/2+p(c-1)+b,(b-2)(b-1)/2+a,p (p-1)/2+q (q-1)/2+p(c-1)+a,1},{p (p-1)/2+q (q-1)/2+p(c-1)+a,(b-2)(b-1)/2+a,p (p-1)/2+q (q-1)/2+p(c-1)+b,-1}},{1}],Apply[{Table[{#1,#2,#3}->#4,{c,p},{b,q},{a,b-1}],Table[{#1,#3,#2}->-#4,{c,p},{b,q},{a,b-1}]}&,{{p (p-1)/2+q (q-1)/2+p(a-1)+c,p (p-1)/2+(b-2)(b-1)/2+a,p (p-1)/2+q (q-1)/2+p(b-1)+c,-1},{p (p-1)/2+q (q-1)/2+p(b-1)+c,p (p-1)/2+(b-2)(b-1)/2+a,p (p-1)/2+q (q-1)/2+p(a-1)+c,1}},{1}],(* --- *)Apply[{Table[If[#4!=0,{#1,#2,#3}->#4,Nothing],{a,p},{b,q},{c,q}],Table[If[#4!=0,{#1,#3,#2}->-#4,Nothing],{a,p},{b,q},{c,q}]}&,{{p (p-1)/2+(Max[b,c]-1)(Max[b,c]-2)/2+Min[b,c],p (p-1)/2+q (q-1)/2+p(b-1)+a,p (p-1)/2+q (q-1)/2+p(c-1)+a,Which[b<c,-1,b>c,1,b==c,0]}},{1}],Apply[{Table[If[#4!=0,{#1,#2,#3}->#4,Nothing],{a,p},{b,q},{c,p}],Table[If[#4!=0,{#1,#3,#2}->-#4,Nothing],{a,p},{b,q},{c,p}]}&,{{(Max[a,c]-1)(Max[a,c]-2)/2+Min[a,c],p (p-1)/2+q (q-1)/2+p(b-1)+a,p (p-1)/2+q (q-1)/2+p(b-1)+c,Which[a<c,-1,a>c,1,a==c,0]}},{1}]]],{p (p-1)/2+q (q-1)/2+p q,p (p-1)/2+q (q-1)/2+p q,p (p-1)/2+q (q-1)/2+p q}]]/;p>=1&&q>=1


LieMatricesSL[n_Integer]:=Module[{i,j},SparseArray[Flatten[Join[Table[If[i<n||j<n,{n(i-1)+j,i,j}->1,Nothing],{i,n},{j,n}],Table[{(n+1)(i-2)+1,i,i}->-1,{i,2,n}]]]]]/;n>=2


LieMatricesSO[n_Integer]:=Module[{i,j},SparseArray[Flatten[Join[Table[{(j-1)(j-2)/2+i,i,j}->-1,{j,n},{i,j-1}],Table[{(j-1)(j-2)/2+i,j,i}->1,{j,n},{i,j-1}]]],{n (n-1)/2,n,n}]]/;n>=2


LieMatricesSU[n_Integer]:=Module[{i,j,k},SparseArray[Flatten[Join[Apply[Table[{#1,#2,#3}->#4,{j,n},{i,j-1}]&,{{(j-1)^2+2i-1,i,j,-I/2},{(j-1)^2+2i-1,j,i,-I/2},{(j-1)^2+2i,i,j,-1/2},{(j-1)^2+2i,j,i,1/2}},{1}],Apply[Table[{#1,#2,#3}->#4,{k,n-1},{j,k}]&,{{k^2,j,j,-I/Sqrt[2k(k+1)]}},{1}],Apply[Table[{#1,#2,#3}->#4,{j,2,n}]&,{{(j-1)^2,j,j,I Sqrt[(j-1)/(2j)]}},{1}]]],{n^2-1,n,n}]]/;n>=2


LieMatricesSp[n_Integer]:=Module[{i,j},SparseArray[Flatten[Join[Table[{n(i-1)+j,i,j}->1,{i,n},{j,n}],Table[{n(i-1)+j,n+j,n+i}->-1,{i,n},{j,n}],Table[{n^2+j (j-1)/2+i,i,n+j}->1,{j,n},{i,j-1}],Table[{n^2+j (j-1)/2+i,j,n+i}->1,{j,n},{i,j-1}],Table[{n^2+j (j+1)/2,j,n+j}->1,{j,n}],Table[{n (3n+1)/2+j (j-1)/2+i,n+i,j}->1,{j,n},{i,j-1}],Table[{n (3n+1)/2+j (j-1)/2+i,n+j,i}->1,{j,n},{i,j-1}],Table[{n (3n+1)/2+j (j+1)/2,n+j,j}->1,{j,n}]]],{n(2n+1),2n,2n}]]/;n>=1


LieMatricesTri[n_Integer]:=Module[{i,j},SparseArray[Flatten[Table[{j (j-1)/2+i,i,j}->1,{j,n},{i,j}]],{n (n+1)/2,n,n}]]/;n>=1


LieAlgSuTri[n_Integer]:=Module[{i,j,k},SparseArray[Flatten[Apply[{Table[{#1,#2,#3}->#4,{k,n},{j,k-1},{i,j-1}],Table[{#1,#3,#2}->-#4,{k,n},{j,k-1},{i,j-1}]}&,{{(k-2) (k-1)/2+i,(j-2) (j-1)/2+i,(k-2) (k-1)/2+j,1}},{1}]],{n (n-1)/2,n (n-1)/2,n (n-1)/2}]]/;n>=2


LieMatricesSOpq[p_Integer,q_Integer]:=SparseArray[Flatten[Join[Table[{(j-2)(j-1)/2+i,i,j}->-1,{j,p},{i,j-1}],Table[{(j-2)(j-1)/2+i,j,i}->1,{j,p},{i,j-1}],
Table[{p (p-1)/2+(j-2)(j-1)/2+i,p+i,p+j}->-1,{j,q},{i,j-1}],Table[{p (p-1)/2+(j-2)(j-1)/2+i,p+j,p+i}->1,{j,q},{i,j-1}],Table[{p (p-1)/2+q (q-1)/2+p(j-1)+i,i,p+j}->1,{i,p},{j,q}],Table[{p (p-1)/2+q (q-1)/2+p(j-1)+i,p+j,i}->1,{i,p},{j,q}]]],{p (p-1)/2+q (q-1)/2+p q,p+q,p+q}]/;p>=1&&q>=1


LieMatricesExcG2={DiagonalMatrix[{0,1,-1,0,-1,1,0}],DiagonalMatrix[{0,0,1,-1,0,-1,1}],SparseArray[{{2,3}->1,{6,5}->-1},{7,7}],SparseArray[{{3,2}->1,{5,6}->-1},{7,7}],SparseArray[{{2,4}->1,{7,5}->-1},{7,7}],SparseArray[{{4,2}->1,{5,7}->-1},{7,7}],SparseArray[{{3,4}->1,{7,6}->-1},{7,7}],SparseArray[{{4,3}->1,{6,7}->-1},{7,7}],SparseArray[{{1,2}->Sqrt[2],{5,1}->-Sqrt[2],{3,7}->-1,{4,6}->1},{7,7}],SparseArray[{{2,1}->-Sqrt[2],{1,5}->Sqrt[2],{7,3}->1,{6,4}->-1},{7,7}],SparseArray[{{1,3}->Sqrt[2],{6,1}->-Sqrt[2],{2,7}->1,{4,5}->-1},{7,7}],SparseArray[{{3,1}->-Sqrt[2],{1,6}->Sqrt[2],{7,2}->-1,{5,4}->1},{7,7}],SparseArray[{{1,4}->Sqrt[2],{7,1}->-Sqrt[2],{2,6}->-1,{3,5}->1},{7,7}],SparseArray[{{4,1}->-Sqrt[2],{1,7}->Sqrt[2],{6,2}->1,{5,3}->-1},{7,7}]};


LieAlgExcG2=SparseArray[Transpose[Outer[LinearSolve[Flatten[LieMatricesExcG2,{{2,3},{1}}],Flatten[#1.#2-#2.#1]]&,LieMatricesExcG2,LieMatricesExcG2,1],{2,3,1}]];


LieAlgExcE[n_]:=(* de Graaf \[Section]5.13 p.189 *) Module[{i,j,i0,j0,k0,m0,i2,j2,k2,w1,w2,ctrMtrx1,posRoots,dim,canonicalVectors,additionTable,asymCoefficients,asymFct},ctrMtrx1=SparseArray[Join[Table[{i,i-1}->-1,{i,3,n}],Table[{i-1,i}->-1,{i,3,n}],{{1,4}->-1,{4,1}->-1,{i_,i_}->2}],{n,n}];posRoots=LieCartanMtrx2RootSystem[ctrMtrx1];
additionTable=Flatten[Table[{i,j,FirstPosition[posRoots,Part[posRoots,i]+Part[posRoots,j]]}/.{{i2_,j2_,_Missing}->Nothing,{i2_,j2_,{k2_}}->{i2,j2,k2}},{i,Length[posRoots]},{j,Length[posRoots]}],1];
canonicalVectors=posRoots.ctrMtrx1;dim=n+2 Length[posRoots];asymFct[w1_,w2_]:=(-1)^(w1.w2+Part[w1,1]Part[w2,4]+Drop[Drop[w1,3],-1].Drop[w2,4]+Take[w1,{3,4}].Take[w2,{2,3}]);
asymCoefficients[{i2_,j2_,k2_}]:={(* (i) *) If[i2<j2,{n+2 k2-1,n+2 i2-1,n+2 j2-1}->asymFct[Part[posRoots,i2],Part[posRoots,j2]],Nothing],(* (ii) *) If[i2<j2,{n+2 k2,n+2 i2,n+2 j2}->asymFct[-Part[posRoots,i2],-Part[posRoots,j2]],Nothing], (* (iii) *) {n+2 j2-1,n+2 k2-1,n+2 i2}->asymFct[Part[posRoots,i2]+Part[posRoots,j2],-Part[posRoots,i2]], (* (iv) *) {n+2 j2,n+2 i2-1,n+2 k2}->asymFct[Part[posRoots,i2],-Part[posRoots,i2]-Part[posRoots,j2]]};SparseArray[Flatten[Join[Flatten[Table[{n+2 j-1,i,n+2 j-1}->Part[posRoots,j,i],{i,n},{j,Length[posRoots]}]],Flatten[Table[{n+2 j,i,n+2 j}->-Part[posRoots,j,i],{i,n},{j,Length[posRoots]}]],Flatten[Table[If[Part[canonicalVectors,i,k]!=0,{k,n+2i,n+2i-1}->Part[canonicalVectors,i,k],Nothing],{i,Length[canonicalVectors]},{k,n}]],Flatten[asymCoefficients/@additionTable]]/.{({i0_,j0_,k0_}->m0_)->{{i0,j0,k0}->m0,{i0,k0,j0}->-m0}}],{dim,dim,dim}]]/;MemberQ[{6,7,8},n]


LieAlgExcF4:=(* de Graaf \[Section]5.15. We depend heavily on the details of the construction of E6, in particular the direction of the edges used in defining the asymmetry function for the structure constants must be compatible with the diagram automorphism. *)Module[{x=LieAlgExcE[6],ctrMtrx1=SparseArray[Join[Table[{i,i-1}->-1,{i,3,6}],Table[{i-1,i}->-1,{i,3,6}],{{1,4}->-1,{4,1}->-1,{i_,i_}->2}],{6,6}],diagramAutomorphism=Cycles[{{2,6},{3,5}}],posRoots,rank,numPRoots,rootPermutation,totalPermutation,totalPermMatrix,newBase,baseOfComplement,projOntoFixspace,k},posRoots=LieCartanMtrx2RootSystem[ctrMtrx1];rank=Length[ctrMtrx1];numPRoots=Length[posRoots];rootPermutation=FindPermutation[posRoots,Permute[#,diagramAutomorphism]&/@posRoots];totalPermutation=FindPermutation[Range[rank+2numPRoots],Join[Permute[Range[rank],diagramAutomorphism],Flatten[Permute[Table[{rank+2k-1,rank+2k},{k,numPRoots}],rootPermutation]]]];totalPermMatrix=SparseArray[MapIndexed[({First[#2],#1}->1)&,PermutationList[totalPermutation,rank+2numPRoots]],{rank+2numPRoots,rank+2numPRoots}];newBase=NullSpace[totalPermMatrix-sparseIdentityMatrix[rank+2numPRoots]];baseOfComplement=NullSpace[totalPermMatrix+sparseIdentityMatrix[rank+2numPRoots]];projOntoFixspace=Take[#,Length[newBase]]&/@Inverse[Join[newBase,baseOfComplement]];SparseArray[Transpose[projOntoFixspace].(Transpose[Transpose[x,{1,3,2}].Transpose[newBase],{1,3,2}].Transpose[newBase])]];


LieAxiomCheck[x_]:=ArrayQ[x]&&((Length[#]==1&&Part[#,1,2]==3)&[Tally[Dimensions[x]]]) && Simplify[x==-Transpose/@x]&&
Simplify[(#+Transpose[#,{2,3,1,4}]+Transpose[#,{3,1,2,4}]==ConstantArray[0,ConstantArray[Length[x],4]])&[Dot[Transpose[#,{3,1,2}],Transpose[#,{3,1,2}]]&[x]]];


LieCenter[x_]:=NullSpace[Flatten[x,{{1,2},{3}}]];


LieKillingForm[x_]:=Tr[Transpose[x.x,{1,3,4,2}],Plus,2];


LieIsNilpotent[x_]:=Module[{y=sparseIdentityMatrix[Length[x]],previousRank=Length[x]+1,n=Length[x]},While[Length[y]!=0&&Length[y]<previousRank,previousRank=Length[y];y=Select[RowReduce[Flatten[x.Transpose[y],{{2,3},{1}}]],(#!=ConstantArray[0,n])&]];Length[y]==0];


LieIsCartanAlgebra[liealg_,subsp_]:=Module[{isSelfNormalizing,isNilpotent},isSelfNormalizing[liealg0_,subsp0_]:=Module[{n=Length[liealg0],subsp1, annihilator,subrank,coefftest},subsp1=Select[RowReduce[subsp0],(#!=ConstantArray[0,n])&];If[subsp1=={},Return[n==0]];subrank=Length[subsp1];annihilator=NullSpace[subsp1];If[Length[annihilator]==0,Return[True]];coefftest= Select[RowReduce[Flatten[annihilator.liealg0.Transpose[subsp1],{{1,3},{2}}]],(#!=ConstantArray[0,n])&];If[Length[coefftest]!=n-subrank,Return[False]];coefftest.Transpose[subsp1]==ConstantArray[0,{n-subrank,subrank}]];isNilpotent[liealg2_,subsp2_]:=Module[{y=subsp2,previousRank=Length[subsp2]+1,n=Length[liealg2]},While[Length[y]!=0&&Length[y]<previousRank,previousRank=Length[y];y=Select[RowReduce[Flatten[Transpose[liealg2.Transpose[y],{1,3,2}].Transpose[subsp2],{{2,3},{1}}]],(#!=ConstantArray[0,n])&]];Length[y]==0];isSelfNormalizing[liealg,subsp]&&isNilpotent[liealg,subsp]];


LieFindCartanSubalgebra[liealg_]:=Module[{subalg0=sparseIdentityMatrix[Length[liealg]],nullSpace4Lie,isNilpotent,nonNilpotentElement,candidate0,candidate1,rank0,rankLie=Length[liealg],dummy,subalg2,loopctr},nullSpace4Lie[x_]:=Module[{y=x,n=Length[x],m=1},While[m<=n,y=x.x;m=2m];NullSpace[y]];isNilpotent[x2_]:=Module[{y2=x2,n2=Length[x2],m2=1},If[Tr[x2]!=0,False,While[m2<=n2&&y2!=SparseArray[{},{n2,n2}],y2=y2.y2;m2=2m2];y2==SparseArray[{},{n2,n2}]]];nonNilpotentElement[liealg1_,subalg1_]:=Module[{x1,n1,m1=1,transform,dualbase=Transpose[subalg1].Inverse[subalg1.Transpose[subalg1]]},x1=Join[subalg1,Flatten[MapIndexed[If[Part[#2,1]<Part[#2,2],Plus@@#1,Nothing]&,Outer[{#1,#2}&,subalg1,subalg1,1],{2}],1]];n1=Length[x1];transform=x1.Transpose[liealg1,{2,3,1}];While[m1<=n1&&isNilpotent[Transpose[dualbase].Part[transform,m1].Transpose[subalg1]],m1=m1+1];If[m1<=n1,Extract[x1,{{m1}}],{}]];candidate0=nonNilpotentElement[liealg,subalg0];
If[candidate0=={},Return[subalg0]];candidate0=Part[candidate0,1];
subalg0=nullSpace4Lie[liealg.candidate0];
rank0=Length[subalg0];
For[loopctr=0,loopctr<=rankLie,loopctr++,candidate1=nonNilpotentElement[liealg,subalg0];
If[candidate1=={},Return[subalg0]];candidate1=Part[candidate1,1];
For[dummy=0,dummy<=rankLie+1,dummy++,subalg2=nullSpace4Lie[liealg.(candidate0+dummy(candidate1-candidate0))];
If[Length[subalg2]<rank0,candidate0=candidate0+dummy(candidate1-candidate0);subalg0=subalg2;rank0=Length[subalg2];Break[]]]];
];


JordanBlockDecompose[w_]:=Module[{z=JordanDecomposition[w],v,s1,j1,s2},v=Function[x,Join@@(Part[#,2]&/@x)]/@GatherBy[MapIndexed[{#1,#2}&,Diagonal[Part[z,2]]],Part[#,1]&];s1=Extract[Inverse[Part[z,1]],#]&/@Partition[#,1]&/@v;j1=Extract[Map[Extract[#],Part[z,2],1],#]&/@(Partition[#,1]&/@v);
s2=Map[Extract[#],Part[z,1],1]&/@(Partition[#,1]&/@v);Transpose[{s2,j1,s1}]];


LieReprRootDecomposition[liealg_,csa_,repr_]:=Module[{z,u={{sparseIdentityMatrix[Length[Part[repr,1]]],{},sparseIdentityMatrix[Length[Part[repr,1]]]}},subdivide,i},subdivide[{i0_,rv0_,p0_},x0_]:=Module[{u0=JordanBlockDecompose[p0.x0.i0],f},f[{i1_,j1_,p1_}]:={i0.i1,Append[rv0,Part[j1,1,1]],p1.p0};f/@u0];
z=csa.repr;For[i=1,i<=Length[z],i++,u=Flatten[subdivide[#,Part[z,i]]&/@u,1]];u];


LieRootDecomposition[x_]:=Module[{z0=LieFindCartanSubalgebra[x],v,isNullVector,tripletransf},isNullVector[x2_]:=And@@Map[(#==0)&,x2,{1}];tripletransf[{i2_,j2_,p3_}]:=If[isNullVector[j2],{Transpose[z0],j2,Inverse[p3.Transpose[z0]].p3},{i2,j2,p3}];v=LieReprRootDecomposition[x,z0,Transpose[x,{2,1,3}]];tripletransf/@v];


LieChevalleyBase[x_,csa_]:=Module[{u=LieReprRootDecomposition[x,csa,Transpose[x,{2,1,3}]],kf=LieKillingForm[x],isNullVector,tripletransf,cart0,rootVectors,biForm,cartanMatrix,firstBasis,dummy,dummy1,tmp,fbExpand,rootBase,collectedRootSystem,pullSign,dynkinGraph,rootGraphAux0,rootGraphAux1,rStrip,canonicalRootSystem={},posRootVector,negRootVector,prevRootVector,baseRootVector,cart1,permuteMe},isNullVector[x2_]:=And@@Map[(#==0)&,x2,{1}];tripletransf[{i2_,j2_,p3_}]:=If[isNullVector[j2],{Transpose[csa],j2,Inverse[p3.Transpose[csa]].p3},{i2,j2,p3}];
pullSign[x5_]:=If[Or@@((#>0)&/@x5),{x5,1},{-x5,-1}];
u=tripletransf/@u;cart0=Part[Select[u,isNullVector[Part[#,2]]&],1];rootVectors=Extract[2]/@Select[u,(!isNullVector[Part[#,2]])&];biForm=FullSimplify[Inverse[Transpose[Part[cart0,1]].kf.Part[cart0,1]]];
For[firstBasis={};dummy=1,(Length[firstBasis]<Length[biForm])&&(dummy<=Length[rootVectors]),dummy++,tmp=Append[firstBasis,Part[rootVectors,dummy]];If[Length[NullSpace[Transpose[tmp]]]==0,firstBasis=tmp]];
fbExpand=Select[LinearSolve[Transpose[firstBasis]]/@rootVectors,(FirstCase[#,t_/;t!=0,-1]>0)&];
fbExpand=Complement[fbExpand,Flatten[Table[Part[fbExpand,dummy]+Part[fbExpand,dummy1],{dummy,Length[fbExpand]},{dummy1,dummy}],1]];rootBase=FullSimplify[fbExpand.firstBasis];
biForm=FullSimplify[rootBase.biForm.Transpose[rootBase]];
cartanMatrix=Transpose[Inner[Times,(2/#)&/@Diagonal[biForm],biForm,List]];
collectedRootSystem=(*{root form, positive vector, negative vector}*){Part[#,1,1],If[Part[#,1,2]==1,Part[#,1,3],Part[#,2,3]],If[Part[#,1,2]==1,Part[#,2,3],Part[#,1,3]]}&/@Gather[FullSimplify[If[isNullVector[Part[#,2]],Nothing,Append[pullSign[LinearSolve[Transpose[rootBase],Part[#,2]]],First[Transpose[Part[#,1]]]]]&/@u],(First[#1]==First[#2])&];
rootGraphAux0=Sort[Gather[MapIndexed[{First[#2],Total[First[#1]],First[#1]}&,collectedRootSystem],(Part[#1,2]==Part[#2,2])&],(Part[#1,1,2]<Part[#2,1,2])&];
tmp=DirectedEdge[0,#]&/@(Extract[1]/@First[rootGraphAux0]);
rootGraphAux1=Graph[Join[tmp,If[Length[rootGraphAux0]!=1,Apply[DirectedEdge,Flatten[Table[Outer[If[And@@(Function[x6,x6>=0]/@(Part[#2,3]-Part[#1,3])),{Part[#1,1],Part[#2,1]},Nothing]&,Part[rootGraphAux0,dummy],Part[rootGraphAux0,dummy+1],1],{dummy,Length[rootGraphAux0]-1}],{1,2,3}],{1}],{}]]];rootGraphAux1=Drop[First[Last[Reap[DepthFirstScan[rootGraphAux1,0,{"DiscoverVertex"->(Sow[{#2,#1}]&)}]]]],1];For[dummy=1,dummy<=Length[rootGraphAux1],dummy++,posRootVector=Part[collectedRootSystem,Part[rootGraphAux1,dummy,2],2];If[Part[rootGraphAux1,dummy,1]!=0,prevRootVector=Part[canonicalRootSystem,First[FirstPosition[rootGraphAux1,{_,Part[rootGraphAux1,dummy,1]}]]];baseRootVector=Part[canonicalRootSystem,First[FirstPosition[Take[Extract[1]/@canonicalRootSystem,Length[rootBase]],Part[collectedRootSystem,Part[rootGraphAux1,dummy,2],1]-Part[collectedRootSystem,Part[rootGraphAux1,dummy,1],1]]]];
tmp=Simplify[(Conjugate[posRootVector].((x.Part[prevRootVector,2]).Part[baseRootVector,2]))/(Conjugate[posRootVector].posRootVector)];(* Humphreys p.147 Thm(Chevalley) part (d) *)rStrip=1;While[True,If[(Total[Part[prevRootVector,1]]<=rStrip)||!MemberQ[Extract[3]/@Part[rootGraphAux0,Total[Part[prevRootVector,1]]-rStrip],Part[prevRootVector,1]-rStrip Part[baseRootVector,1]],Break[]];rStrip++];posRootVector*=(tmp/rStrip)];negRootVector=Part[collectedRootSystem,Part[rootGraphAux1,dummy,2],3];tmp=Simplify[2(Conjugate[posRootVector].posRootVector)/(Conjugate[posRootVector].((x.posRootVector).(x.negRootVector).posRootVector))];negRootVector*=tmp;AppendTo[canonicalRootSystem,{Part[collectedRootSystem,Part[rootGraphAux1,dummy,2],1],Simplify[posRootVector],Simplify[negRootVector]}]];
permuteMe=FindPermutation[Extract[1]/@Select[canonicalRootSystem,(Total[Part[#,1]]==1)&],IdentityMatrix[Length[cartanMatrix]]];cart1=Permute[Simplify[((x.Part[#,3]).Part[#,2])&/@Select[canonicalRootSystem,(Total[Part[#,1]]==1)&]],permuteMe];tmp=Transpose[Join[cart1,Flatten[Drop[#,1]&/@canonicalRootSystem,1]]];dynkinGraph=Graph[Range[Length[cartanMatrix]],Flatten[Table[Switch[Part[cartanMatrix,dummy,dummy1]Part[cartanMatrix,dummy1,dummy],0,Nothing,1,UndirectedEdge[dummy,dummy1],2,If[Abs[Part[cartanMatrix,dummy1,dummy]]==1,DirectedEdge[dummy1,dummy,2],DirectedEdge[dummy,dummy1,2]],3,If[Abs[Part[cartanMatrix,dummy1,dummy]]==1,DirectedEdge[dummy1,dummy,3],DirectedEdge[dummy,dummy1,3]]],{dummy,Length[cartanMatrix]},{dummy1,dummy-1}]],EdgeLabels->"EdgeTag",VertexLabels->"Name"];
{Diagonal[biForm],cartanMatrix,dynkinGraph,cart1,canonicalRootSystem,SparseArray[Simplify[Inverse[tmp].(Transpose[Transpose[x,{1,3,2}].tmp,{1,3,2}].tmp)]]}];


LieChevalleyBase[x_]:=LieChevalleyBase[x,LieFindCartanSubalgebra[x]];


LieCartanMtrx2RootSystem[ctrMtrx_]:=Module[{u={{1,IdentityMatrix[Length[ctrMtrx]]}},heighestRoots,newRoots,i,j,candidate0,candidate1,stripLength,isNullVector},isNullVector[x2_]:=And@@Map[(#==0)&,x2,{1}];While[True,heighestRoots=Last[u];newRoots={Part[heighestRoots,1]+1,{}};For[i=1,i<=Length[Part[heighestRoots,2]],i++,candidate0=Part[heighestRoots,2,i];For[j=1,j<=Length[ctrMtrx],j++,If[isNullVector[Drop[candidate0,{j}]],Continue[]];stripLength=Part[ctrMtrx,j].candidate0;candidate1=UnitVector[Length[ctrMtrx],j];
If[((stripLength<0)||(stripLength>Part[candidate0,j])||((stripLength<Part[heighestRoots,1]-1)&&MemberQ[Part[u,Part[heighestRoots,1]-stripLength-1,2],candidate0-(stripLength+1) candidate1]))&&(!MemberQ[Part[newRoots,2],candidate0+candidate1]),AppendTo[Part[newRoots,2],candidate0+candidate1]]]];If[Part[newRoots,2]=={},Break[]];AppendTo[u,newRoots]];Flatten[Extract[2]/@u,1]];


LieRepresentationCheck[liealg_,repr_]:=Transpose[liealg,{3,1,2}].repr==Outer[(#1.#2-#2.#1)&,repr,repr,1];


End[]


EndPackage[]
