(* ::Package:: *)

(* :Author: Bernd Guenther *)
(* :Package Version: 2.1 *)
(* :Mathematica Version: 12.1 *)
(* :History: 
   2.0 Eliminated dependency on Combinatorica package. 
   2.1 Added eigenvalues *)
   
BeginPackage["IrrCharSymGrp`"];


DecrPartitionQ::usage="DecrPartitionQ[\[Lambda]] is true if and only if \[Lambda] is an integer partition, arranged in decreasing order."


CycleZ::usage="CycleZ[\[Rho]_?DecrPartitionQ] returns the size of the centralizer class of an element of cycle type \[Rho]."


CharacterSymGrp::usage="CharacterSymGrp[\[Lambda]_?DecrPartitionQ,\[Rho]_?DecrPartitionQ] returns the value of the irreducible character \!\(\*SuperscriptBox[\(\[Chi]\), \(\[Lambda]\)]\) evaluated at the conjugacy class \[Rho]."


CompoundYoungChar::usage="CompoundYoungChar[\[Lambda]_?DecrPartitionQ] returns the composite Young character corresponding to the partition \[Lambda]."


CharTblSymGrp::usage="CharTblSymGrp[n_Integer] returns the character table of the symmetric group. Each character is stored as a row vector."


ExtCharTblSymGrp::usage="ExtCharTblSymGrp[n_Integer] returns the extended character table of the symmetric group, i.e. the multiplicities of primitive roots of unity."


KostkaMatrix::usage="KostkaMatrix[n_Integer] returns the matrix of Kostka numbers of rank n."


CharacterScalarProduct::usage="CharacterScalarProduct[f_List,g_List,groupRank_Integer] returns the scalar product of the class vectors f and g."


NTranspDecomp::usage="NTranspDecomp[pi_?PermutationListQ] represents pi as product of transpositions of immediate neighbors. An entry value of k in the returned list denotes the transposition (k,k+1).\[IndentingNewLine]Attention: Permutations are multiplied right to left like right operators, not like functions!"


ExpandNTrDecom::usage="ExpandNTrDecom[ntr_List] is the inverse operation of NTranspDecomp."


InvariantYMetric::usage="InvariantYMetric[\[Lambda]_?DecrPartitionQ] is the scalar product invariant under Young's natural presentation corresponding to the integer partition \[Lambda]."


CoxeterTest::usage="CoxeterTest[ynr_] applied to the matrices of Young's natural representation checks whether these matrices satisfy Coxeter's relations, as they must. So unless you tamper with the definitions this function should always return TRUE."


YnrCharacterTest::usage="YnrCharacterTest[ynr_,\[Lambda]_] applied to the matrices of Young's natural representation corresponding to the integer partition \[Lambda] computes the character and compares it to the relevant entry in the character table. So unless you tamper with the definitions this function should always return TRUE. A complete test would be for instance: \[IndentingNewLine]testPartition=RandomPartition[5];\[IndentingNewLine]testYnr=YoungsNaturalRepresentation[testPartition];\[IndentingNewLine]CoxeterTest[testYnr]&&YnrCharacterTest[testYnr,testPartition]"


WeakLeftBruhatGraph::usage="WeakLeftBruhatGraph[\[Lambda]_?DecrPartitionQ] Construct weak left Bruhat graph of standard tableaux.
Start with rowwise ordered tableau (observe that it is smallest with respect to weak left Bruhat ordering) and then do breadth first algorithm.
Output is a record with two components; first is the list of stanard tableaux.
Second is the list of weighted edges, where weight i means that the transposition (i,i+1) the first connected tableau to the second. The edges are given as three component record with the first two components denoting the indices of the connected tableaux and the third record the weight."


Seminormal2Natural::usage="Seminormal2natural[\[Lambda]_?DecrPartitionQ] The transformation matrix turning the seminormal presentation into the natural presentation. Each row vector is the expansion of a natural basis vector in terms of the seminormal basis vectors."


NormSquareOfTableau::usage="NormSquareOfTableau[myTableau_] computes the norm squares of the seminormal basis vectors."


YoungsNaturalRepresentation::usage="YoungsNaturalRepresentation[\[Lambda]_?DecrPartitionQ] computes the matrices of Young's natural representation of the symmetric group corresponding to the integer partition \[Lambda] by transforming the seminormal representation. The function returns the images of the transpositions of immediate neighbors, listed in order of the transposed elements. The matrices are supposed to operate from the right on row vectors."


YoungsSeminormalRepresentation::usage="YoungsSeminormalRepresentation[\[Lambda]_?DecrPartitionQ] computes the matrices of Young's seminormal representation of the symmetric group corresponding to the integer partition \[Lambda]. The function returns the images of the transpositions of immediate neighbors, listed in order of the transposed elements. The matrices are supposed to operate from the right on row vectors."


YoungsNaturalReprValue::usage="YoungsNaturalReprValue[\[Lambda]_?DecrPartitionQ,pi_?PermutationListQ] is the matrix assigned to permutation \[Pi] by Young's natural representation corresponding to partition \[Lambda]."


YoungsSeminormalReprValue::usage="YoungsSeminormalReprValue[\[Lambda]_?DecrPartitionQ,pi_?PermutationListQ] is the matrix assigned to permutation \[Pi] by Young's seminormal representation corresponding to partition \[Lambda]."


Begin["`Private`"]


DecrPartitionQ[\[Lambda]_]:=VectorQ[\[Lambda],(IntegerQ[#]&&(#>0))&]&&Apply[And,Map[(#<=0)&,Differences[\[Lambda]]]];


transposeMe[\[Lambda]_?DecrPartitionQ]:=Table[First[Last[Position[\[Lambda],x_/;x>=y]]],{y,Part[\[Lambda],1]}]; (* Computes the transpose of a partition. *)


hookFormula[\[Lambda]_?DecrPartitionQ]:=
With[{\[Mu]=transposeMe[\[Lambda]]},Factorial[Total[\[Lambda]]]/Product[\[Lambda][[j]]-i+\[Mu][[i]]-j+1,{j,Length[\[Lambda]]},{i,\[Lambda][[j]]}]]; (* Computes the number of tableaux on a partition using the hook length formula. *)


nextPartition[\[Lambda]_?DecrPartitionQ]:=Module[{i=First[Last[Position[\[Lambda],x_/;x>1]]],j,k=Length[\[Lambda]],l},j=Part[\[Lambda],i];l=QuotientRemainder[j+k-i,j-1];
Join[Take[\[Lambda],i-1],ConstantArray[j-1,Part[l,1]],If[Part[l,2]>=1,{Part[l,2]},{}]]]/;AnyTrue[\[Lambda],#>1&];


nextPartition[\[Lambda]_?DecrPartitionQ]:={Total[\[Lambda]]}/;AllTrue[\[Lambda],#==1&]; (* Convenction: at the last partition we cycle back to the first one. *)


transposeTableau[t_]:=Module[{transshape=transposeMe[Length/@t]},Table[Part[t,j,i],{i,Length[transshape]},{j,Part[transshape,i]}]]; (* Transposition of a Young tableau; substitute for the corresponding combinatorica function. *)


permInversions[pi_?PermutationListQ]:=Sum[If[Part[pi,i]>Part[pi,j],1,0],{j,Length[pi]},{i,j-1}]; (* The number of inversions in a permutation; substitute for the corresponding combinatorica function. *)


CharacterSymGrp[{},{}]:=1; 


CharacterSymGrp[\[Lambda]_?DecrPartitionQ,\[Rho]_?DecrPartitionQ]:=
hookFormula[\[Lambda]]/;
Total[\[Lambda]]==Total[\[Rho]]&&Length[\[Rho]]>=1&&First[\[Rho]]==1;


CharacterSymGrp[\[Lambda]_?DecrPartitionQ,\[Rho]_?DecrPartitionQ]:=
With[{
\[Rho]max=First[\[Rho]],\[Rho]0=Drop[\[Rho],1],
\[Mu]=transposeMe[\[Lambda]],
\[Nu]=Append[\[Lambda],0]},
Sum[If[\[Lambda][[j]]+\[Mu][[i]] +1-j-i!=\[Rho]max,0,If[EvenQ[\[Rho]max+i-\[Lambda][[j]]],-1,1] * 
CharacterSymGrp[
Select[Table[If[k<j||\[Nu][[k]]<i,\[Nu][[k]],Max[\[Nu][[k+1]],i]-1],{k,Length[\[Lambda]]}],(#>0)&],\[Rho]0] ],
{j,1,Length[\[Lambda]]},{i,1,\[Lambda][[j]]}]]/;
Total[\[Lambda]]==Total[\[Rho]]&&Length[\[Rho]]>=1&&First[\[Rho]]>1;


CycleZ[\[Rho]_?DecrPartitionQ]:=Apply[Times,Map[(Factorial[#[[2]]])&,Tally[\[Rho]]]] * Apply[Times,\[Rho]];


CompoundYoungChar[\[Lambda]_?DecrPartitionQ]:=Module[{chrVect=Table[0,{PartitionsP[Total[\[Lambda]]]}],
supPartitionTupel=Partition[\[Lambda],1],
hashPositionTupel=Prime[\[Lambda]],
r,columnIdx},
With[{hashPosList=SparseArray[Flatten[MapIndexed[{Times@@Prime[#1]->First[#2]}&,IntegerPartitions[Total[\[Lambda]]]]]]},
While[True,
columnIdx=Part[hashPosList,Times@@hashPositionTupel];
chrVect[[columnIdx]]+=Times@@Apply[Multinomial,Map[(Part[#,2])&,SplitBy[Sort[Flatten[Tally/@supPartitionTupel,1]],First],{2}],2];
r=Length[supPartitionTupel];
While[(r>0)&&(First[supPartitionTupel[[r]]]==1),r--];
If[r<=0,Break[]];
supPartitionTupel=Join[Take[supPartitionTupel,r-1],{nextPartition[supPartitionTupel[[r]]]},Partition[Drop[\[Lambda],r],1]];
hashPositionTupel=Join[Take[hashPositionTupel,r-1],{Times@@Prime[supPartitionTupel[[r]]]},Prime[Drop[\[Lambda],r]]]
]];
chrVect];


CharTblSymGrp[n_Integer]:=With[{weights=CycleZ/@IntegerPartitions[n],rDimension=PartitionsP[n]},
Module[{resTbl=CompoundYoungChar/@ IntegerPartitions[n],
denomVector,kostkaNumbers},
For[k=2,k<=rDimension,k++,
denomVector=resTbl[[k]]/weights;
kostkaNumbers= (Dot[denomVector,#])&/@Take[resTbl,k-1];
resTbl[[k]]-=Dot[kostkaNumbers, Take[resTbl,k-1]];
];resTbl]]/;n>0


ExtCharTblSymGrp[n_Integer]:=Module[{stdCharTbl=CharTblSymGrp[n], cycleTypes=IntegerPartitions[n],cycleTpAssoc,getListOfMultiples,cycleGraph,processThisPartition,elemOrders,calcOrders,eigenvmultipl,primePowerVect,u2,v2,w2,l,i,faux},faux[(p_Integer->l_List)]:={ConstantArray[p,Length[l]],l};getListOfMultiples[aPartition_,idx_]:=Function[power,DirectedEdge[First[idx],First[Lookup[cycleTpAssoc,Key[ReverseSort[Flatten[If[Divisible[#,power],ConstantArray[#/power,power],#]&/@aPartition]]]]],power]]/@DeleteCases[Union@@((Extract[1]/@FactorInteger[#])&/@aPartition),1];
calcOrders[DirectedEdge[u1_,v1_,w1_]]:=(elemOrders[[v1]]={w1 Part[elemOrders,u1,1] (* order of element *),If[Divisible[Part[elemOrders,u1,1],w1],w1 ,w1-1] Part[elemOrders,u1,2] (* EulerPhi of Order *)});
processThisPartition[partidx_]:=Module[{r1vector=Part[elemOrders,partidx,2](Extract[partidx]/@stdCharTbl),calcMutiplicities1,calcMutiplicities2},calcMutiplicities1[DirectedEdge[u1_,v1_,w1_]]:=(r1vector+= Part[elemOrders,v1,2] (Extract[v1]/@stdCharTbl));
calcMutiplicities2[DirectedEdge[u1_,v1_,w1_]]:=Module[{k0},k0=Part[elemOrders,partidx,1]/Part[elemOrders,u1,1];
AppendTo[Part[eigenvmultipl,partidx],k0 w1->If[Divisible[k0,w1],Part[eigenvmultipl,primePowerVect[{partidx,w1}]][k0]/w1,(* *)(Part[eigenvmultipl,primePowerVect[{partidx,w1}]][k0]-Part[eigenvmultipl,partidx][k0])/(w1-1)(* *)]]];
DepthFirstScan[cycleGraph,partidx,{"FrontierEdge"->calcMutiplicities1}];r1vector/=Part[elemOrders,partidx,1];AppendTo[Part[eigenvmultipl,partidx],1->r1vector];DepthFirstScan[cycleGraph,partidx,{"FrontierEdge"->calcMutiplicities2}]];
elemOrders=Append[ConstantArray[{},Length[cycleTypes]-1],{1,1}];cycleTpAssoc=PositionIndex[cycleTypes];cycleGraph=If[n==1,Graph[{1},{}],Graph[Flatten[MapIndexed[getListOfMultiples,cycleTypes]]]];primePowerVect=Association[EdgeList[cycleGraph]/.DirectedEdge[u2_,v2_,w2_]->({u2,w2}->v2)];
eigenvmultipl=ConstantArray[Association[],Length[cycleTypes]];
BreadthFirstScan[ReverseGraph[cycleGraph],Length[cycleTypes],{"PostvisitVertex"->processThisPartition,"FrontierEdge"->calcOrders}];Transpose[DeleteCases[Transpose[#,{2,3,1}]&/@Map[faux,Normal/@KeySort/@eigenvmultipl,{2}],{i_Integer,0},{3}]]]/;n>0


KostkaMatrix[n_Integer]:=CharTblSymGrp[n] . DiagonalMatrix[(1/CycleZ[#])&/@IntegerPartitions[n]] . Transpose[CompoundYoungChar/@ IntegerPartitions[n]]/;n>0


CharacterScalarProduct[f_List,g_List,groupRank_Integer]:=Dot[f*g,(1/CycleZ[#])&/@IntegerPartitions[groupRank]]/;(groupRank>0)&&(Length[f]==PartitionsP[groupRank])&&(Length[g]==PartitionsP[groupRank])


NTranspDecomp[pi_?PermutationListQ]:=
Module[{idx=1,transList={},pi2=pi},While[idx<Length[pi2],
If[pi2[[idx]]<pi2[[idx+1]],idx++,
transList=Append[transList,idx];
pi2=System`Permute[pi2,System`Cycles[{{idx,idx+1}}]];
If[idx>1,idx--,idx++]]];transList];


ExpandNTrDecom[ntr_List]:=PermutationList[Apply[PermutationProduct,System`Cycles[{{#,#+1}}]&/@ntr]];


CoxeterTest[ynr_]:=And[Apply[And,(#.#==IdentityMatrix[Length[ynr[[1]]]])&/@ynr],
And@@Table[ynr[[r]].ynr[[r+1]].ynr[[r]]==ynr[[r+1]].ynr[[r]].ynr[[r+1]],{r,Length[ynr]-1}],
And@@Flatten[Table[ynr[[r]].ynr[[s]]==ynr[[s]].ynr[[r]],
{r,Length[ynr]-2},{s,r+2,Length[ynr]}]]];


cTypeRepresentative[\[Lambda]_?DecrPartitionQ]:=
Flatten[Apply[Range,Transpose[{Prepend[Drop[#,-1]+1,1],#-1}],{1}]]&[Accumulate[\[Lambda]]];


YnrCharacterTest[ynr_,\[Lambda]_]:=(Append[Tr/@Apply[Dot,Extract[ynr,#]&/@Partition[#,1]&/@cTypeRepresentative/@Drop[IntegerPartitions[Total[\[Lambda]]],-1],{1}],Length[ynr[[1]]]]
==Part[CharTblSymGrp[Total[\[Lambda]]],Part[Position[IntegerPartitions[Total[\[Lambda]]],\[Lambda]],1,1]]);


InvariantYMetric[\[Lambda]_?DecrPartitionQ]:=
With[{wlbg1=WeakLeftBruhatGraph[\[Lambda]],
transform=Seminormal2Natural[\[Lambda]]},
Times@@Factorial/@transposeMe[\[Lambda]]transform.DiagonalMatrix[NormSquareOfTableau/@First/@wlbg1].Transpose[transform]];


predPermutations1[invPList_,curPos_,sourcePos_]:=MapIndexed[{System`Permute[invPList,First[#1]],{curPos,sourcePos,Last[#1]}}&,{System`Cycles[{{#,#+1}}],#}&/@Flatten[Position[Differences[invPList],x_/;x<0]]];


predPermutations2[invPListList_,curPos_,sourcePos_]:=
MapIndexed[{First[Part[#1,1]],Function[x,ReplacePart[x,1->Part[x,1]+First[#2]]]/@Part[#1,2]}& ,
Transpose/@Gather[
Join@@MapIndexed[predPermutations1[#1,curPos,sourcePos+First[#2]]&,
First/@invPListList],
(Part[#1,1]==Part[#2,1])&]];


rowWiseInvPList[\[Lambda]_?DecrPartitionQ]:=
PermutationList[System`InversePermutation[PermutationCycles[Join@@transposeTableau[(Range@@#)&/@Drop[FoldList[{1+Last[#1],#2+Last[#1]}&,{0,0},\[Lambda]],1]]]],Total[\[Lambda]]];


WeakLeftBruhatGraph[\[Lambda]_?DecrPartitionQ]:=
With[{x=rowWiseInvPList[\[Lambda]],n=Total[\[Lambda]],
shape=Drop[FoldList[{1+Last[#1],#2+Last[#1]}&,{0,0},transposeMe[\[Lambda]]],1]},
Function[v,{transposeTableau[Function[w,Take[PermutationList[System`InversePermutation[PermutationCycles[Part[v,1]]],n],w]]/@shape],
Part[v,2]}]/@
Flatten[Nest[Append[#,predPermutations2[
Last[#],Length[Flatten[#,1]],Length[Flatten[#,1]]-Length[Last[#]]]]&,
{{{x,{}}}},permInversions[x]],1]];


youngAuxiliary[\[Lambda]_,modus_]:=
(* modus=1: only transform; modus=2: natural presentation; modus=3: seminormal presentation *)
With[{wlbg1=WeakLeftBruhatGraph[\[Lambda]]},
Module[{wlbgAdjacencyLists,contentVectors,spanningTree,transform,tnorm,tinv,semimatrix},
(* The following expression computes the adjacency lists of the weak left Bruhat graph;an entrySubscript[a, ij]may have four different meanings,depending on the following cases:i) IfSubscript[a, ij]=ithen j and j+1 are contained in the same row of tableau i.ii) IfSubscript[a, ij]=-ithen j and j+1 are contained in the same column of tableau i.iii) IfSubscript[a, ij]\[NotEqual]\[PlusMinus]ibutSubscript[a, ij]<0then i and i+1 appear inverted in tableau i and application of the admissible transposition (j,j+1) turns tableau i into tableauSubscript[a, ij],thus removing an inversion.iv) IfSubscript[a, ij]\[NotEqual]\[PlusMinus]ibutSubscript[a, ij]>0then i and i+1 appear in correct order in tableau i and application of the admissible transposition (j,j+1) turns tableau i into tableauSubscript[a, ij],thus adding an inversion. *)
wlbgAdjacencyLists=SparseArray[
(({Part[#,1],Part[#,3]}->Part[#,2])&/@ Flatten[Part[#,2]&/@wlbg1,1])
~Join~
(({Part[#,2],Part[#,3]}->-Part[#,1])&/@ Flatten[Part[#,2]&/@wlbg1,1])
~Join~
Flatten[MapIndexed[Function[{v,w},Function[u,{First[w],u}->First[w]]/@v],Function[v,Last/@Select[Transpose[{Flatten[Function[u,Append[u,0]]/@(Differences/@v)],Flatten[v]}],Function[u,First[u]==1]]]/@(First/@wlbg1)]]
~Join~
Flatten[MapIndexed[Function[{v,w},Function[u,{First[w],u}->-First[w]]/@v],Function[v,Last/@Select[Transpose[{Flatten[Function[u,Append[u,0]]/@(Differences/@v)],Flatten[v]}],Function[u,First[u]==1]]]/@(transposeTableau[#]&/@(First/@wlbg1))]],{Length[wlbg1],Total[\[Lambda]]-1}]
;
contentVectors=Function[u,SparseArray[Flatten[MapIndexed[Function[{v1,v2},{v1->Last[v2]-First[v2]}],First[u],{2}]],{Total[\[Lambda]]}]]/@wlbg1;
If[modus!=3,
	spanningTree=If[Length[wlbg1]==1,{},First/@MapIndexed[Drop[#1/.(Rule[{a_},b_]):>{First[#2],a,b},-1]&,ArrayRules/@SparseArray[Flatten[Function[v,Function[u,{Part[u,2],Part[u,1]}->Part[u,3]]/@Last[v]]/@Drop[wlbg1,1]],{Length[wlbg1]-1,Length[wlbg1]}]]];
	transform=SparseArray[{Length[wlbg1],Length[wlbg1]}->1,{Length[wlbg1],Length[wlbg1]}];
	Module[{k,r,s,x},For[i=Length[spanningTree],i>0,i--,
	(* e_i = s_r e_k is the base vector to be constructed. *)
	k=Part[spanningTree,i,2];
	r=Part[spanningTree,i,3];
	For[j=k,j<=Length[wlbg1],j++,
	(* v_j is a Young vector appearing in e_k with coefficient x. *)
	x=Part[transform,k,j];
	If[x==0,Continue[]];
	s=Part[wlbgAdjacencyLists,j,r];
	Switch[s,
	(* row inversion *) j,Part[transform,i,j]+=x,
	(* column inversion *) -j,Part[transform,i,j]-=x,
	(* removing an inversion *) x_/;x<0,Part[transform,i,j]+=x/(Part[contentVectors,j,r+1]-Part[contentVectors,j,r]);
	Part[transform,i,-s]+= x (1-1/(Part[contentVectors,j,r+1]-Part[contentVectors,j,r])^2),
	(* admissibly adding an inversion *)_,Part[transform,i,s]+=x;
	Part[transform,i,j]+=x/(Part[contentVectors,j,r+1]-Part[contentVectors,j,r]);
	]]]];
	tnorm=transform];
If[modus==1,Return[tnorm]];
semimatrix=SparseArray[Flatten[Module[{s},Table[s=Part[wlbgAdjacencyLists,k,r];
Switch[s,
(* row inversion *) k,{{r,k,k}->1},
(* column inversion *) -k,{{r,k,k}->-1},
(* removing an inversion *) x_/;x<0,
{{r,k,k}->1/(Part[contentVectors,k,r+1]-Part[contentVectors,k,r]),
{r,k,-s}->1-1/(Part[contentVectors,k,r+1]-Part[contentVectors,k,r])^2},
(* admissibly adding an inversion *)_,
{{r,k,k}->1/(Part[contentVectors,k,r+1]-Part[contentVectors,k,r]),
{r,k,s}->1}],
{r,Total[\[Lambda]]-1},{k,Length[wlbg1]}]]],
{Total[\[Lambda]]-1,Length[wlbg1],Length[wlbg1]}];
If[modus==3,Return[semimatrix]];
tinv=Inverse[tnorm];
tnorm.#.tinv&/@ semimatrix
]];


Seminormal2Natural[\[Lambda]_?DecrPartitionQ]:=youngAuxiliary[\[Lambda],1]/; Total[\[Lambda]]>1


YoungsNaturalRepresentation[\[Lambda]_?DecrPartitionQ]:=youngAuxiliary[\[Lambda],2]/; Total[\[Lambda]]>1


YoungsSeminormalRepresentation[\[Lambda]_?DecrPartitionQ]:=youngAuxiliary[\[Lambda],3]/; Total[\[Lambda]]>1


NormSquareOfTableau[myTableau_]:=With[{trshape=transposeMe[Length/@myTableau]},
Product[If[((i2>i1)\[Or](j2>j1))\[And](Part[myTableau,i1,j1]>Part[myTableau,i2,j2]),
1-1/(i1-j1-i2+j2)^2,1],
{j1,1,Length[trshape]},{i1,1,Part[trshape,j1]},
{j2,j1,Length[trshape]},{i2,1,Part[trshape,j2]}]];


YoungsNaturalReprValue[\[Lambda]_?DecrPartitionQ,pi_?PermutationListQ]:=If[pi==Range[Total[\[Lambda]]],IdentityMatrix[hookFormula[\[Lambda]]],Dot@@Extract[YoungsNaturalRepresentation[\[Lambda]],Partition[NTranspDecomp[pi],1]]]/;Total[\[Lambda]]==Length[pi];


YoungsSeminormalReprValue[\[Lambda]_?DecrPartitionQ,pi_?PermutationListQ]:=If[pi==Range[Total[\[Lambda]]],IdentityMatrix[hookFormula[\[Lambda]]],Dot@@Extract[YoungsSeminormalRepresentation[\[Lambda]],Partition[NTranspDecomp[pi],1]]]/;Total[\[Lambda]]==Length[pi];


End[]


EndPackage[]
