(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 9.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       157,          7]
NotebookDataLength[     63268,       1250]
NotebookOptionsPosition[     61252,       1187]
NotebookOutlinePosition[     62441,       1222]
CellTagsIndexPosition[     62355,       1217]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell[TextData[{
 "LatticePlanes",
 " (",
 ButtonBox["Crystallica",
  BaseStyle->"Link",
  ButtonData->"paclet:Crystallica/guides/Crystallica"],
 " package option)"
}], "Section"],

Cell[BoxData[
 StyleBox["\<\"LatticePlanes is an option of CrystalPlot which gives a list \
of lattice planes to include in the plot.\"\>", "MSG"]], "Print", "PrintUsage",
 CellTags->"Info3576394248-7634371"],

Cell[TextData[{
 "LatticePlanes",
 " is an option of ",
 ButtonBox["CrystalPlot",
  BaseStyle->"Link",
  ButtonData->"paclet:Crystallica/ref/CrystalPlot"],
 "."
}], "Item"],

Cell["\<\
LatticePlanes gives a list of planes in the following form: \
{{{h,k,l},dist},{{h,k,l},dist},...}, where \[OpenCurlyQuote]dist\
\[CloseCurlyQuote] is the distance of the plane to the origin in reduced \
length. Planes are automatically extended or truncated for cell shape and \
supercell size.\
\>", "Item"],

Cell["\<\
In hexagonal systems, people like to use 4 Miller indices instead of 3, i.e. \
[h,k,i,l] instead of [h,k,l]. The index i is redundant in this notation, and \
for the LatticePlanes option, you would still specify only h, k, and l.\
\>", "Item"],

Cell["\<\
For plane styling, use the option ContourStyle (default: \
ContourStyle\[Rule]{Red,Opacity[.5]}; for plane edges, use BoundaryStyle \
(default: BoundaryStyle\[Rule]None). Other options of ContourPlot3D will also \
be recognized.\
\>", "Item"],

Cell["\<\
For 2D plots, the format of the planes is {{{h,k},dist},{{h,k},dist},...}, \
and the default for ContourStyle is {Thick,Red}. Other options of ContourPlot \
will also be recognized.\
\>", "Item"],

Cell["Default:  {}", "Item"],

Cell[CellGroupData[{

Cell["Examples", "Subsection"],

Cell["Load Crystallica:", "Text"],

Cell[BoxData[
 RowBox[{"Needs", "[", "\"\<Crystallica`\>\"", "]"}]], "Input",
 CellContext->"Global`"],

Cell["The {1,1,1}-plane in a cubic system:", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CrystalPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"3", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "3", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "3"}], "}"}]}], "}"}], ",", 
   RowBox[{"LatticePlanes", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", "1"}], "}"}], 
     "}"}]}]}], "]"}]], "Input",
 CellContext->"Global`"],

Cell[BoxData[
 Graphics3DBox[{{
    {GrayLevel[0], 
     TubeBox[{{{0, 0, 0}, {3, 0, 0}, {3, 3, 0}, {0, 3, 0}, {0, 0, 0}}, {{0, 0,
       3}, {3, 0, 3}, {3, 3, 3}, {0, 3, 3}, {0, 0, 3}}}, 0.02]}, 
    {GrayLevel[0], 
     TubeBox[{{{0, 0, 0}, {0, 0, 3}}, {{3, 0, 0}, {3, 0, 3}}, {{3, 3, 0}, {3, 
      3, 3}}, {{0, 3, 0}, {0, 3, 3}}}, 0.02]}}, {
    {GrayLevel[0], 
     TubeBox[{{0, 0, 0}, {2.4980039840795545`, 0., 0.}, {2.4980039840795545`, 
       0., 0.}, {3, 0, 0}}, {0.021, 0.021, 0.12549900398011132`, 0}]}, 
    {GrayLevel[0], 
     TubeBox[{{0, 0, 0}, {0., 2.4980039840795545`, 0.}, {0., 
       2.4980039840795545`, 0.}, {0, 3, 0}}, {
      0.021, 0.021, 0.12549900398011132`, 0}]}, 
    {GrayLevel[0], 
     TubeBox[{{0, 0, 0}, {0., 0., 2.4980039840795545`}, {0., 0., 
       2.4980039840795545`}, {0, 0, 3}}, {
      0.021, 0.021, 0.12549900398011132`, 0}]}}, 
   GraphicsComplex3DBox[CompressedData["
1:eJyF2muMldUVxvHDXGBQNIFJK6KxrQhYtF6p1qIeSiPQglWpNaKWgPdb66Vg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    "], {
     {RGBColor[1, 0, 0], Opacity[0.5], EdgeForm[None], 
      StyleBox[GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJwt1mdUFkcUh/FdY2zYO3YssYEVe4kFG6IoVkRRASOIgl3EbuxdsWFXUOwa
e+/H9G56TEwziYmJSYypJj73zP/D77z7zpnduXPn7swGxaVGpeTyPK+A73lP
8WvX9stfbzqmYTlmYjaWYgVWYg3mqn0GFmIxFqh9HuajGqqgBgqhGAagIiqr
zxikIlnjDkY0YhCr69FIwVQkqT0eCRir61WKzWKehTmKzZ4/CiNQFZVQHrVR
S7E1VDzl1Kc66qEMSqEE6qMBGqEkeqo9Cv3QS/17IEJzrKKxCiCvcltceSiC
osiD3MpVTQRpLay9ICoo5hDk1xqVVT4Lq09dBKOO4umrPCdp7okYiiEYiJYE
0gRTMQezMRMzMByxmIw0rMIsPId4rMAIxGENWmEaFmMJliOdMVZrvZbyfxmi
MBejkKSxbNzpvsv/RNXYFP5PxDiEopP6WJzpinmKridZXNwzUutb356le9NU
M1ZXaXr+ONXSNOVkvI2n9hjlZ5jaJ2AyhiNO7enqPwlPa03t+i7uWZvys1J5
CEMd1Pbde5SqNdqNTOzT+k5Q7VmdBCBDNVZaMdRXvVkNLOJZgxGN/lrrSAzy
3LuzDIdxCAfRHTuwHUfxAo5gK7ZhC3YhCzsV2x5kYxM2ozH244BiPo4TOIZT
OIOTCMW/6Ib/8D8e4wrO4Sz+UZ/HuveE4rmMCziP2/gMn6r/X/hTcW5Tn72K
80O8hJu4iou4hNcUs+XhFfRGH5zGW5rLy3gDb+J1vI1OyMENvIhryuc7eBUf
4D28izB0UK6s/WN8hHVYj7V6zk3FsFH32rrcwid6Tnu8jw2eqytbd9sTfsW3
+Bs/oCvaaE3DlWfb9+xdy9BzLJ52GitTz7S1/sNzNXBLYz1CW/yu/t/jC3yu
eVkeOqrPsxrX9ivbD23vCvTc/tMCXRRbZzRDczRFS8/Vcz7P7Xut0cpzddLE
c3V1XOtrtfFQcT7SOlp+rD4fqL89P7edV/Bxn/+2qf7ITy7fxWkx38HPmovN
6yd8pz52r72nDzX337ReVnvHNJdwtVvO7+neu1qLbzxXqzs8V5O3lTdr/8pz
dfsLvsaXnqvhB4rB4uzA2AvQFHmRD3m0xy7RPhyAQiiLQBRGfhRBeZRDRQSr
T6j288bqY/faud4d3dAZxVAcRVEQFdSnOqqisuIppT4Bis3GrYsyijMI1VAF
JdW/BNpY/tHTd3Nsh0jFEIFwPSdY8UzQPj8eB3AQ+9EAbVEP7dFc8wrTWdBR
7dantcZtp2vb/+08CkFpxdwDQzEMY1ETtfAMuio/XTRuI7RAM41ra2RnZUPF
cxhHcEj5r4QavtvrbJ+0OcZgCAb5rgZsb7GzMsF352a84rHzNFZxDlMObY3s
vEhSbu0strNjoO/OzWitndVDX9+9X7aO9j1m49q5MNJ3Z4HFYGe9nfuJvnsf
+/HbX7EFqh76+O67boDvzo6B6tNHtRGhOdq3h9Wevcu2l9o5YudOkOrH8mnn
i+3zVhtXPLf/W13ZXmE5t++Wmsqb1bCdX1ZLtofc1zubrHUZrbW7o/dojO/i
H6C5b9K7b2uXov7Juk7VWifo2u7dorWzGuuF3r77JjmKvdiNHGRjJ07YHHAR
Z3ASp7AHWbiBCziH0+pvfY5jH3ZgG7YjExssZqzHWsz33T5g/Tf67t2x2tvl
uzoMUZwWQ6T6L8Q839Xn8757F+y7y77T7Ftote++PTKwTnO09/EsrinOqziv
uVzCZc3R5pKo+pmr58/T3G2O1xVblvKzSPFcV/zZmqPleSs2K4ZMzTdHed6s
nGxVfp4AeKleKQ==
          "]], Polygon3DBox[{{138, 41, 132, 131}, {40, 41, 138, 136}}], 
         Polygon3DBox[{{132, 130, 125, 129, 131}, {39, 35, 37, 40, 38}, {38, 
          40, 136, 135, 134}}], Polygon3DBox[CompressedData["
1:eJwt1mXYF0UUhvFZQkBSwqJBVFpCkQ5p9JVUQpESASlBaQERVCRFFBQEKWkp
FZWQsrs7sDsxse7nOs+H38Xyf2d3Z8/MOWcqDhjVdWSulNJMnIAKKItZ/q0q
quAHfIQRWUqjMBJX4wF+24VH8Ci2YxsexkO+3oEu2I2P8SkGcO+VGJTFM9di
PZbhHqzEErzC2PuxBa/jTbyNNViNPdiLO7AYd/p6Z4q5aQ4PYj/2YTlW+N5V
vvcx/IE//S2692/URz0sw7/4x2OO+/o/HPC3X8Zcb8FhPIq9uArl+VsdzMF0
zHBsdzhWFRhTGZdwfTFyc50Hf3H9BT5HXv5/lH/fx3uOodbiwxQxnI95uBXT
MBD38rdO6OBvOe5vHOWYa8xgXIHRuAmtcDEOaF2xCwnfpVjr4WiPdliA27AQ
EzDbc3iRsS/gdo/R3LTWQxyT3TiEBow5D+dy3Qi10AwtUQ/X8rcJeq/mjKmO
od41DmM0BuNxXRZ7YymW4A28hldTjBmLyWiM2rgG/XE9pqANWqOqY6IYTsV0
j+mLMiiE7rhU38WzF+Ed70m9613vsefwBJ7C09iMTXgGT+KIr7d6bz+Lgmjk
Zy7AQsUQ1Ty3C3AjZmCoc2Q5NmELNmIFJjkm+sY6ju1E39sA53mtZ2EmiuMk
P3MYeqIbzsZZOB2lUYC5nIrLue7jva01vYLfeqaI1Q1+l9a0vmPbBXdgMe7E
HHTN4tsVk424y2u3xdfrsBat0NJr2px7lmZRE+7yM3OyyLWH8RD24BHvsfu4
ZwPWow1a+5laux7oncW39Oa3ua4J92BlitxRTVBteRyHcdBrqrU7hEscq15Z
5H4fP7NfFjnV13usJEooV7ym8zAZwzHRe+MlvJVizusck2Fea63RzVnUFr1L
e2O+53wZpviZyp1z0DCLWDVFE60Ff/sR36bI5btT1DTVLtW0jvx2Ic5AebTw
vTleu85Z1ITuKWpUQxRBYcVS88RQDMRoDFZNSlGLZjs3J3lvKKe6pegFqnV6
Zld0xliMwTUpcl8183ScjLy4DuM9ZlyKOXdCW9RAWVTxdXWcgPwoiFOxDfdj
B3Zis3NHOaKaUBN1HUPVoo68ox3apoiheoFqaYcsauBFOMXv0hzOz6J2NXEM
WznXLuKeU1ACOaiNumiC4iiKUh5zofIrRU6pNp6P/uiXojfniiVM+RyTPMid
Ijfzp6ghJ3qMauYIr8VVqI5qXrvTUuRybj+zEMqhjGOunqUzQGmc5LUu5DkU
85zPQCWPqYDKqJgihquxKouevtR7so1rQj3HSrVINU29VXtSPaWD17Sjc1xn
ANW0e7VvXUPOcQxres6aQy3HsJhjoj1T0jHviz6O4QAM8l5doxqBddiQxdlD
c1ateFkxzKIX59I8U9TAojgxi17QyDFXLisHe/ldyoUhKXr3z/gJH6To3Z+l
6K3KKdXAudiPfajkHNQeLpdF7T0zixxXjerpvdHH7/oeX6U4A+hsph6tHNdZ
oqLvVQ3XWjf0ntHeUE9RLZqUIqc0t0/wfIrerbOZzhgjU9Qofcsw7yX14vb4
Bl96DqotypHpKXq09lgNTMP1eDCLs4S+8TEczKJWF9Hcsug1qqWqyeqtX6c4
8yhWR/2NelepLGqperR6onpWMed4Pl8X9zMLZFEz1eN0htEc1CN0Nhvvb5/o
b9QZQzUnx3tGtWWsc0F7rLBrSAHPuZD3QPMUeyOv56yYNE1RE07OouaX8txU
i3T2WIQjWey91a45WmvVLtWowt5j6rk6Sx9IUZMreU11VtQZTGdvndn6+V2q
RTp76MzzG35P0ct01v0VzTw3rZ1q2i8pzkgt0cJjjvneHilqsmq7cl81qrGf
ecz3qjbqbKleqR6t3qocae536ZnKqe3YmkXt1VlFZ/v/ARxWXeU=
          "]]}],
       Lighting->Automatic]}, {}, {}, {}, {}},
    VertexNormals->CompressedData["
1:eJztWcsNwjAMDXSMAlJnYAYWYAQkzpy7BVswBhcmYRK49FIp8qdO8uLk9A6N
nOfnFztRp9vjet+HED67EIY/Hs7Py2v+vqV4XKE2jnZfqzi5+beOi94jCB+u
z7Q+QcnXi8+1esbyH4nvKIjOz4r/KVIfNB0pH3L3seozqepF8UHtb2g6luah
ra/1ufCC0vO9RqpvxOrDjdOxLqT6F0of4frSOn+tLii6SXlJeWvjoerTKtZa
j+Xezp1Hrcwv7TuAux51TqT2eS151sLfam533Kaj136Ifr9tTf/UOnjRE+09
TvHh1nvrvSR1vmj9QeqH0jzRkPKR1X0QTX/rvoGWX2ldU62vHa3fna3pl6su
XdeO3Vf5+Er/x3nBUn5A9SEqL2/4A6XRpQM=
     "]], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{0, 0, 0}, 0],
       "Ball"]},
     "1"],
    Annotation[#, 1, "Tooltip"]& ]},
  BaseStyle->{
    Specularity[
     GrayLevel[0.5], 100]},
  Boxed->False,
  Lighting->"Neutral",
  SphericalRegion->True]], "Output"]
}, Open  ]],

Cell["\<\
Several {1,1,1}-planes in a periodically repeated cubic supercell with \
user-defined layout:\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CrystalPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"3", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "3", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "3"}], "}"}]}], "}"}], ",", 
   RowBox[{"Sysdim", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"2", ",", "2", ",", "2"}], "}"}]}], ",", 
   RowBox[{"CellLineStyle", "\[Rule]", "2"}], ",", 
   RowBox[{"LatticePlanes", "\[Rule]", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", "dist"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"dist", ",", "1", ",", "5"}], "}"}]}], "]"}]}], ",", 
   RowBox[{"ContourStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<TerreVerte\>\"", ",", 
      RowBox[{"Opacity", "[", ".7", "]"}]}], "}"}]}], ",", 
   RowBox[{"BoundaryStyle", "\[Rule]", "Thick"}]}], "]"}]], "Input",
 CellContext->"Global`"],

Cell[BoxData[
 Graphics3DBox[{{
    {GrayLevel[0], TubeBox[{{0, 0, 0}, {0, 0, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 3, 0}, {0, 3, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 6, 0}, {0, 6, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{3, 0, 0}, {3, 0, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{3, 3, 0}, {3, 3, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{3, 6, 0}, {3, 6, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{6, 0, 0}, {6, 0, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{6, 3, 0}, {6, 3, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{6, 6, 0}, {6, 6, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 0, 0}, {6, 0, 0}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 0, 3}, {6, 0, 3}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 0, 6}, {6, 0, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 3, 0}, {6, 3, 0}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 3, 3}, {6, 3, 3}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 3, 6}, {6, 3, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 6, 0}, {6, 6, 0}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 6, 3}, {6, 6, 3}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 6, 6}, {6, 6, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 0, 0}, {0, 6, 0}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{3, 0, 0}, {3, 6, 0}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{6, 0, 0}, {6, 6, 0}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 0, 3}, {0, 6, 3}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{3, 0, 3}, {3, 6, 3}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{6, 0, 3}, {6, 6, 3}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{0, 0, 6}, {0, 6, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{3, 0, 6}, {3, 6, 6}}, 0.02]}, 
    {GrayLevel[0], TubeBox[{{6, 0, 6}, {6, 6, 6}}, 0.02]}}, {
    {GrayLevel[0], 
     TubeBox[{{0, 0, 0}, {2.4980039840795545`, 0., 0.}, {2.4980039840795545`, 
       0., 0.}, {3, 0, 0}}, {0.021, 0.021, 0.12549900398011132`, 0}]}, 
    {GrayLevel[0], 
     TubeBox[{{0, 0, 0}, {0., 2.4980039840795545`, 0.}, {0., 
       2.4980039840795545`, 0.}, {0, 3, 0}}, {
      0.021, 0.021, 0.12549900398011132`, 0}]}, 
    {GrayLevel[0], 
     TubeBox[{{0, 0, 0}, {0., 0., 2.4980039840795545`}, {0., 0., 
       2.4980039840795545`}, {0, 0, 3}}, {
      0.021, 0.021, 0.12549900398011132`, 0}]}}, 
   GraphicsComplex3DBox[CompressedData["
1:eJx1lllIVGEYhsdxXLINGyo3KjM1ymyzfZmhxYKMzEjSFsqIigRLtKtKBVso
KlqkRInqwmiPLMooaDMtbdGEsYUKA2ccrfSQzZTNMS96vgNn8GIYzjPv+Zb3
//7/n8iMrJRNRoPBsL/n49vzmRk3Z2xFH8Wyvn/JsWd/PBbFVGB++bh0bqCz
sPaup6NX/q3f6eQaf1X41JjpZ8xmh+X11v6BJ1QfK/zNuaovg7rsvfKTLYc2
F5mMVn2cmEXh5s1ul6X4Xp5vQ7Pdgh6emHDb8E7V+M3EkPptJrfoiUP9/lV7
6y5V20QP33ivveaWW+P0hR5/yAvHBzhx4NSDPta3bLTD4xJ92P3Zb5uUHOEv
Fg5Kzo1wi5+fmn6cDRmpWqrrstaNrLRJX/DVudPiku12iV8elrM9qtvHij7i
3LOH1asaLYkRVYVDVY/o4ejX1i5JWb2yUeKgh5P3wfNXC1oitfoybu0qdXS6
hJPXtnjYsos/OyQO+jV531/cCFGkX/Rw9Bs+xJlTXdkSBz2cvOiJcz52RFdT
vEPqoV/qH3vnUfm41AzxnzjUB5+xY2h4q4/WJ+uIHt+IzzrCqVOfl/nX87ak
pVWPu01W/I/6HLR8S6dN9MwV9aCHo2cfEQc9nLzw8tqamEo/P9FTT98Jh/cp
hkDZj8QnL5zv3vTEl7rtrTVboxvFf9YXPfyaX8XLywZFOP6jJw77l77Qwzkf
4Oxr9MTBZ3yA99YveuIfCSq+MitY65M4cOaT3+snF9WmdAeInvmZf6Fw2Kye
fOjh6Jkr3kMPJy+c/avX88w5oO4+oEYFa+cv/sCph/hw5p/3gs5OinYW5ctz
4sQuR7YnwMu/g0lfy3YO1OaF+tFHnLqYviI40Os9uTeu923+6a/9zr67kB1d
ljneYdFznvXnA/PG75wzPDc7k0LaQ92id+aXnAz41eF1nsP15y316PVwfEZP
HP29wDrqz0P8py/uBepHDye+rM//vEfzjtcNDlO8fIBT/5kbVx0VkYrkJT73
HZzzHE6/6Dln0N+vLC2Z/NfHyj6FP/mYu2WMryr7Fx/QUw994Sd54fgDpy/9
/wT01Mn/gZ35p1f0MypeHD3nFX2xXwq6lsx7+9ckfeHv0yEJBtsA7bxA35ZW
XNBg9BM9+4I46P+83pg+pdM7PnHUL4Vp0wZo+xk/Y0c53ycYjVbON3jb8A8t
sS6P9AVn/uH4j37P76uPMvc2Sl7WBT08O/XypvHhqsShL/RqfGjo4PkOycs9
y3qRF866Ex9OX6wLnLzMIRzfWF849eCfvi84PlOP3HP//aF+6sRPOD6gJ6/M
gy4O8fGTeeO+Yz7RwZlPOPegfj/iG3nRw/X7MSdn4qW0bruXHo5vcHxGTxzi
87+LueI95vwfjbeGvg==
    "], {{
      {RGBColor[0.219993, 0.370006, 0.059999], Opacity[0.7], EdgeForm[None], 
       StyleBox[
        GraphicsGroup3DBox[{
          Polygon3DBox[{{2, 3, 1}, {39, 41, 54}, {40, 43, 38}, {20, 36, 18}, {
           35, 22, 19}, {11, 12, 10}, {2, 15, 10}, {5, 6, 4}, {8, 9, 7}, {14, 
           15, 13}, {52, 53, 51}, {49, 50, 48}, {125, 123, 121}, {127, 128, 
           126}, {11, 9, 51}, {5, 12, 54}, {106, 132, 91}, {45, 41, 38}, {21, 
           79, 19}, {69, 70, 68}, {33, 34, 32}, {21, 23, 31}, {30, 28, 25}, {
           60, 63, 58}, {20, 22, 66}, {76, 29, 24}, {59, 61, 71}, {74, 36, 
           65}, {64, 62, 57}, {46, 72, 44}, {35, 73, 67}, {16, 17, 75}, {30, 
           23, 18}, {81, 86, 85}, {27, 82, 25}, {45, 47, 55}, {56, 42, 37}, {
           122, 114, 115}, {81, 79, 77}, {60, 62, 66}, {26, 28, 84}, {83, 80, 
           31}, {78, 87, 77}, {108, 117, 113}, {89, 90, 88}, {111, 109, 
           124}, {16, 87, 13}, {105, 103, 100}, {99, 98, 97}, {108, 109, 
           107}, {78, 80, 94}, {120, 112, 119}, {102, 104, 100}, {39, 53, 
           37}, {95, 96, 94}, {83, 82, 88}, {49, 118, 32}, {99, 117, 116}, {
           89, 114, 113}, {8, 98, 48}, {95, 90, 97}, {27, 29, 115}, {14, 96, 
           7}, {101, 103, 135}, {111, 112, 110}, {134, 118, 116}, {122, 123, 
           124}, {130, 133, 129}, {134, 131, 129}, {105, 93, 110}, {92, 93, 
           91}, {92, 131, 107}, {52, 50, 126}}], 
          Polygon3DBox[{{53, 52, 126, 128, 56, 37}, {50, 49, 32, 34, 127, 
           126}, {123, 125, 119, 112, 111, 124}, {73, 35, 19, 79, 81, 85}, {
           109, 111, 110, 93, 92, 107}, {82, 27, 115, 114, 89, 88}, {86, 81, 
           77, 87, 16, 75}, {98, 99, 116, 118, 49, 48}, {12, 11, 51, 53, 39, 
           54}, {62, 64, 65, 36, 20, 66}, {131, 92, 91, 132, 130, 129}, {87, 
           78, 94, 96, 14, 13}, {90, 89, 113, 117, 99, 97}, {63, 60, 66, 22, 
           35, 67}, {80, 83, 88, 90, 95, 94}, {96, 95, 97, 98, 8, 7}, {79, 21,
            31, 80, 78, 77}, {17, 16, 13, 15, 2, 1}, {6, 5, 54, 41, 45, 55}, {
           28, 26, 24, 29, 27, 25}, {47, 45, 38, 43, 46, 44}, {112, 120, 135, 
           103, 105, 110}, {41, 39, 37, 42, 40, 38}, {118, 134, 129, 133, 33, 
           32}, {29, 76, 121, 123, 122, 115}, {114, 122, 124, 109, 108, 
           113}, {93, 105, 100, 104, 106, 91}, {3, 2, 10, 12, 5, 4}, {117, 
           108, 107, 131, 134, 116}, {23, 30, 25, 82, 83, 31}, {61, 59, 57, 
           62, 60, 58}, {9, 8, 48, 50, 52, 51}, {36, 74, 84, 28, 30, 18}, {22,
            20, 18, 23, 21, 19}, {103, 101, 68, 70, 102, 100}, {15, 14, 7, 9, 
           11, 10}}]}],
        Lighting->Automatic]}, {}, {}, {}, {}}, 
     {Thickness[Large], 
      Line3DBox[{1, 3, 4, 6, 55, 47, 44, 72, 46, 43, 40, 42, 56, 128, 127, 34,
        33, 133, 130, 132, 106, 104, 102, 70, 69, 68, 101, 135, 120, 119, 125,
        121, 76, 24, 26, 84, 74, 65, 64, 57, 59, 71, 61, 58, 63, 67, 73, 85, 
       86, 75, 17, 1}]}},
    VertexNormals->CompressedData["
1:eJxTTMoPSmViYGBoB2JmIJYxnuC6vPLRflkcNC55GRw0seqGO01qOJAavrj4
pLqPWHNg8tI0Di9amU+puwjxyc0vA+0vXOFOa/9Qaj613UFt/5KbToiNBzkS
8wu5+Z3S8oGQOLH5C5c+apWHpKYTcssFUtMJpeUhvcKH1uUUsflioN1LrPup
VU7JofGlCainVjoaauFJ7/SCy52E4ove5Ri94oXUeh3dHACC+iwn
     "]], GraphicsComplex3DBox[CompressedData["
1:eJyF2muMldUVxvFhLnBgYCJMEURjWxGwaL1SrUWdUxrBVqxKrRG1xPu9VSkY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    "], {{
      {RGBColor[0.219993, 0.370006, 0.059999], Opacity[0.7], EdgeForm[None], 
       StyleBox[GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJwt1mdUFkcUh/FdY2zYO3YssYEVe4kFG6IoVkRRASOIgl3EbuxdsWFXUOwa
e+/H9G56TEwziYmJSYypJj73zP/D77z7zpnduXPn7swGxaVGpeTyPK+A73lP
8WvX9stfbzqmYTlmYjaWYgVWYg3mqn0GFmIxFqh9HuajGqqgBgqhGAagIiqr
zxikIlnjDkY0YhCr69FIwVQkqT0eCRir61WKzWKehTmKzZ4/CiNQFZVQHrVR
S7E1VDzl1Kc66qEMSqEE6qMBGqEkeqo9Cv3QS/17IEJzrKKxCiCvcltceSiC
osiD3MpVTQRpLay9ICoo5hDk1xqVVT4Lq09dBKOO4umrPCdp7okYiiEYiJYE
0gRTMQezMRMzMByxmIw0rMIsPId4rMAIxGENWmEaFmMJliOdMVZrvZbyfxmi
MBejkKSxbNzpvsv/RNXYFP5PxDiEopP6WJzpinmKridZXNwzUutb356le9NU
M1ZXaXr+ONXSNOVkvI2n9hjlZ5jaJ2AyhiNO7enqPwlPa03t+i7uWZvys1J5
CEMd1Pbde5SqNdqNTOzT+k5Q7VmdBCBDNVZaMdRXvVkNLOJZgxGN/lrrSAzy
3LuzDIdxCAfRHTuwHUfxAo5gK7ZhC3YhCzsV2x5kYxM2ozH244BiPo4TOIZT
OIOTCMW/6Ib/8D8e4wrO4Sz+UZ/HuveE4rmMCziP2/gMn6r/X/hTcW5Tn72K
80O8hJu4iou4hNcUs+XhFfRGH5zGW5rLy3gDb+J1vI1OyMENvIhryuc7eBUf
4D28izB0UK6s/WN8hHVYj7V6zk3FsFH32rrcwid6Tnu8jw2eqytbd9sTfsW3
+Bs/oCvaaE3DlWfb9+xdy9BzLJ52GitTz7S1/sNzNXBLYz1CW/yu/t/jC3yu
eVkeOqrPsxrX9ivbD23vCvTc/tMCXRRbZzRDczRFS8/Vcz7P7Xut0cpzddLE
c3V1XOtrtfFQcT7SOlp+rD4fqL89P7edV/Bxn/+2qf7ITy7fxWkx38HPmovN
6yd8pz52r72nDzX337ReVnvHNJdwtVvO7+neu1qLbzxXqzs8V5O3lTdr/8pz
dfsLvsaXnqvhB4rB4uzA2AvQFHmRD3m0xy7RPhyAQiiLQBRGfhRBeZRDRQSr
T6j288bqY/faud4d3dAZxVAcRVEQFdSnOqqisuIppT4Bis3GrYsyijMI1VAF
JdW/BNpY/tHTd3Nsh0jFEIFwPSdY8UzQPj8eB3AQ+9EAbVEP7dFc8wrTWdBR
7dantcZtp2vb/+08CkFpxdwDQzEMY1ETtfAMuio/XTRuI7RAM41ra2RnZUPF
cxhHcEj5r4QavtvrbJ+0OcZgCAb5rgZsb7GzMsF352a84rHzNFZxDlMObY3s
vEhSbu0strNjoO/OzWitndVDX9+9X7aO9j1m49q5MNJ3Z4HFYGe9nfuJvnsf
+/HbX7EFqh76+O67boDvzo6B6tNHtRGhOdq3h9Wevcu2l9o5YudOkOrH8mnn
i+3zVhtXPLf/W13ZXmE5t++Wmsqb1bCdX1ZLtofc1zubrHUZrbW7o/dojO/i
H6C5b9K7b2uXov7Juk7VWifo2u7dorWzGuuF3r77JjmKvdiNHGRjJ07YHHAR
Z3ASp7AHWbiBCziH0+pvfY5jH3ZgG7YjExssZqzHWsz33T5g/Tf67t2x2tvl
uzoMUZwWQ6T6L8Q839Xn8757F+y7y77T7Ftote++PTKwTnO09/EsrinOqziv
uVzCZc3R5pKo+pmr58/T3G2O1xVblvKzSPFcV/zZmqPleSs2K4ZMzTdHed6s
nGxVfp4AeKleKQ==
           "]], Polygon3DBox[{{138, 41, 132, 131}, {40, 41, 138, 136}}], 
          Polygon3DBox[{{132, 130, 125, 129, 131}, {39, 35, 37, 40, 38}, {38, 
           40, 136, 135, 134}}], Polygon3DBox[CompressedData["
1:eJwt1mXYF0UUhvFZQkBSwqJBVFpCkQ5p9JVUQpESASlBaQERVCRFFBQEKWkp
FZWQsrs7sDsxse7nOs+H38Xyf2d3Z8/MOWcqDhjVdWSulNJMnIAKKItZ/q0q
quAHfIQRWUqjMBJX4wF+24VH8Ci2YxsexkO+3oEu2I2P8SkGcO+VGJTFM9di
PZbhHqzEErzC2PuxBa/jTbyNNViNPdiLO7AYd/p6Z4q5aQ4PYj/2YTlW+N5V
vvcx/IE//S2692/URz0sw7/4x2OO+/o/HPC3X8Zcb8FhPIq9uArl+VsdzMF0
zHBsdzhWFRhTGZdwfTFyc50Hf3H9BT5HXv5/lH/fx3uOodbiwxQxnI95uBXT
MBD38rdO6OBvOe5vHOWYa8xgXIHRuAmtcDEOaF2xCwnfpVjr4WiPdliA27AQ
EzDbc3iRsS/gdo/R3LTWQxyT3TiEBow5D+dy3Qi10AwtUQ/X8rcJeq/mjKmO
od41DmM0BuNxXRZ7YymW4A28hldTjBmLyWiM2rgG/XE9pqANWqOqY6IYTsV0
j+mLMiiE7rhU38WzF+Ed70m9613vsefwBJ7C09iMTXgGT+KIr7d6bz+Lgmjk
Zy7AQsUQ1Ty3C3AjZmCoc2Q5NmELNmIFJjkm+sY6ju1E39sA53mtZ2EmiuMk
P3MYeqIbzsZZOB2lUYC5nIrLue7jva01vYLfeqaI1Q1+l9a0vmPbBXdgMe7E
HHTN4tsVk424y2u3xdfrsBat0NJr2px7lmZRE+7yM3OyyLWH8RD24BHvsfu4
ZwPWow1a+5laux7oncW39Oa3ua4J92BlitxRTVBteRyHcdBrqrU7hEscq15Z
5H4fP7NfFjnV13usJEooV7ym8zAZwzHRe+MlvJVizusck2Fea63RzVnUFr1L
e2O+53wZpviZyp1z0DCLWDVFE60Ff/sR36bI5btT1DTVLtW0jvx2Ic5AebTw
vTleu85Z1ITuKWpUQxRBYcVS88RQDMRoDFZNSlGLZjs3J3lvKKe6pegFqnV6
Zld0xliMwTUpcl8183ScjLy4DuM9ZlyKOXdCW9RAWVTxdXWcgPwoiFOxDfdj
B3Zis3NHOaKaUBN1HUPVoo68ox3apoiheoFqaYcsauBFOMXv0hzOz6J2NXEM
WznXLuKeU1ACOaiNumiC4iiKUh5zofIrRU6pNp6P/uiXojfniiVM+RyTPMid
Ijfzp6ghJ3qMauYIr8VVqI5qXrvTUuRybj+zEMqhjGOunqUzQGmc5LUu5DkU
85zPQCWPqYDKqJgihquxKouevtR7so1rQj3HSrVINU29VXtSPaWD17Sjc1xn
ANW0e7VvXUPOcQxres6aQy3HsJhjoj1T0jHviz6O4QAM8l5doxqBddiQxdlD
c1ateFkxzKIX59I8U9TAojgxi17QyDFXLisHe/ldyoUhKXr3z/gJH6To3Z+l
6K3KKdXAudiPfajkHNQeLpdF7T0zixxXjerpvdHH7/oeX6U4A+hsph6tHNdZ
oqLvVQ3XWjf0ntHeUE9RLZqUIqc0t0/wfIrerbOZzhgjU9Qofcsw7yX14vb4
Bl96DqotypHpKXq09lgNTMP1eDCLs4S+8TEczKJWF9Hcsug1qqWqyeqtX6c4
8yhWR/2NelepLGqperR6onpWMed4Pl8X9zMLZFEz1eN0htEc1CN0Nhvvb5/o
b9QZQzUnx3tGtWWsc0F7rLBrSAHPuZD3QPMUeyOv56yYNE1RE07OouaX8txU
i3T2WIQjWey91a45WmvVLtWowt5j6rk6Sx9IUZMreU11VtQZTGdvndn6+V2q
RTp76MzzG35P0ct01v0VzTw3rZ1q2i8pzkgt0cJjjvneHilqsmq7cl81qrGf
ecz3qjbqbKleqR6t3qocae536ZnKqe3YmkXt1VlFZ/v/ARxWXeU=
           "]]}],
        Lighting->Automatic]}, {}, {}, {}, {}}, 
     {Thickness[Large], Line3DBox[CompressedData["
1:eJwt0E0rwwEcB/D/yhtZiYOl5ULNS6BEu7NktxHl4iCtzNNhHtphJ1qe8hBK
ttTWsEZGOI3rXgFHJ5+Vw6e+9Tt8v/3CienRVCgIgnE6CdNNF8uskmGFNdIs
sUiEHpJMMcMkCSYYoJ87BSVuqVKhzAbr5MizyzGnnHDFJeecccE+BzywxQJz
zDJPlk22edX9Qotvfvilwy3Eu3zIEU3eeOKRKvfU2vup88wHn3zRoMweBXbI
cU2RG8YYIU6MQYYYpo8ovf+//wOFwzfC
       "]]}},
    VertexNormals->CompressedData["
1:eJztWcsNwjAMDXSMAlJnYAYWYAQkzpy7BVswBhcmYRK49FIp8qdO8uLk9A6N
nOfnFztRp9vjet+HED67EIY/Hs7Py2v+vqV4XKE2jnZfqzi5+beOi94jCB+u
z7Q+QcnXi8+1esbyH4nvKIjOz4r/KVIfNB0pH3L3seozqepF8UHtb2g6luah
ra/1ufCC0vO9RqpvxOrDjdOxLqT6F0of4frSOn+tLii6SXlJeWvjoerTKtZa
j+Xezp1Hrcwv7TuAux51TqT2eS151sLfam533Kaj136Ifr9tTf/UOnjRE+09
TvHh1nvrvSR1vmj9QeqH0jzRkPKR1X0QTX/rvoGWX2ldU62vHa3fna3pl6su
XdeO3Vf5+Er/x3nBUn5A9SEqL2/4A6XRpQM=
     "]], GraphicsComplex3DBox[CompressedData["
1:eJx123mM1tUVxnFmg2GtMAJWq6aAQNE0iluV6ow2AtYFpCat1FIVKy60SIvG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    "], {{
      {RGBColor[0.219993, 0.370006, 0.059999], Opacity[0.7], EdgeForm[None], 
       StyleBox[GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJw113f8V1Mcx/F7rpJUSknj19RQae+lfu1U2ktS+pV2vx/ZREYKITsUZUQo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           "]], 
          Polygon3DBox[{{477, 386, 387, 478}, {566, 571, 574, 565}, {423, 558,
            557, 435}, {444, 445, 451, 450}, {557, 558, 560, 564}, {447, 445, 
           444, 446}, {206, 207, 204, 201}, {522, 386, 477, 521}, {574, 571, 
           573, 572}, {414, 418, 417, 422}, {409, 418, 414, 408}, {400, 238, 
           239, 401}, {211, 207, 206, 210}, {198, 238, 400, 611}}], 
          Polygon3DBox[CompressedData["
1:eJwtz0kOAVEQxvHXEguWbuAEZisu4QLEsOUGEnEA6USMGzYcwN48szG0EGLj
ACJxAP/Sb/FLOlX1quvzpguJvEMp5YETGywwxxor+BnwIYAowngaSo0wxA0v
w65LP4IQgojjS68EN98xPNh5xxUnXPDGBzXmmjDRQx8DtNEy7Fl5c4SFM6bU
59KDhQP21HfY6jySy9S7G6ijYti9mezAUueWG+VWF3LIosxsUeZ1bsk/wwRj
/f//HXpPhjcpJNGlXkVH55FcP7eyOf0=
           "]], Polygon3DBox[CompressedData["
1:eJxFmGW0VkUUhme+C1yDlm4kBEEkpUvQSwuIhKB0S9mFYGOBCiZ2B2IXYC+7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           "]]}],
        Lighting->Automatic]}, {}, {}, {}, {}}, 
     {Thickness[Large], Line3DBox[CompressedData["
1:eJwt0M8rw3EYB/DPuGE2hWw52EbjshwsDv4AaWqtqTmQuZBEXIk/xs1phzVz
wA6O5KJRIz/+EHl9y+HVc3k/P3oyO0eVw1gIoU6KCSbZZ49dDlgWKpBnjiWK
LLDIpcwpZ5xzwTEnFJjnnjtu6dCkRZtxM8boJ0aSYeIMMEqad9kuL3zywTel
vhDWWGGVGhtsRfRtckWDOtusU6ZClSHZQX6jh6gJkowwTY4ZpsgwS55edBdv
dHmgwytf/HDNDU1aPPHMIznrsv///wP7JipV
       "]]}},
    VertexNormals->CompressedData["
1:eJztWssNwjAMrWCMEiRmYAYWYAQkzj2zRbdgDHZhEnrghBQSJ3Ge7fjSd+nH
fnnPsaOebsv1vpumadku+w3Deb08H+/X/MUDEUPj+3phiMSFzicWlyMv362e
H33dav0Uez72Hirfqftr6yFaf442fKk9fm2Yqj+96gEqT9etY45eqPo5GvPP
KGhlvin9zqh1C5W3Nb5/653U/Kj7NaofktaHUeMpjb/2nArNv1Td9447d98r
3R+tnhty58M1782J+HvPYVaRyiO6jlrD1nyi6pjrQAYvPn9g8m7VV1vtQxxl
6k0KcvlIuq+kxUWds2Lr1eo90lHb+XyuzlLzF9d3pfGg1b8pf2k7h5TCs/Z6
1csPXLyh118a5u7Dzneb/HP7GXTc1Py4zuvR+Tny6kZL/+CY51P3bR1vvfRZ
+l+3lfV13f7Xm5U6Ocq6atGzdl1pjx+tx1F5Q/vxA3hPcxw=
     "]], GraphicsComplex3DBox[CompressedData["
1:eJyF2nus13Udx/Fzg3PkAAkHDmapCxQIXQpozizOT5uAgYLoVpqRt0CFBhaw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    "], {{
      {RGBColor[0.219993, 0.370006, 0.059999], Opacity[0.7], EdgeForm[None], 
       StyleBox[GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJxN1mXYFkUYhuFZOkTqo/MDSelO6RRJpRFpCSlpBAWkG+lupBuxsbE7QLG7
EzDhuo+5f/jjfHdn39nZmWeemd3U/qM6j0wTQvgLafEjfsIPGJOEMAp3oBjl
3CiKNXgDq/AWjmMHGiEvcqI65qEaRnD/cAzFecoHsBf3Up6BezAEt2Ms10dy
HO3nDsJgDEzi9f4+n4PlmI9hrq/279T9WMj5EozhfBnHGzjWROUQ752NWVhG
+T4swdvYhUNYgAxI9D8/K7FGfaS8yPfW4jwrMqIhqqKJ66tveu4q37sCZ/jv
ebzv2A4NcexLOb6JxbiL8jQcxmuO8zc4SPkQDmAxjmIE10djutsfj4mYqthx
fZLmj+NJPIHvsYdrp7EbczEPx93PI1iNb6n3mZ97kvIJ17mV8m3oj2OUJyje
2EL5O2zGu3gP74TYn0muozkax7UheBh3U36B44t4CeecS6/73skey9gkxmS6
x3Xe7Z/zfClur+KpEHPjaY5rOa73fJWh3Bil0Q7t0Rb10QD1nLeqUxf5kAcp
GID5mIvymIUKmKaYawxeM1mdJ7m8RpT/3dETHUPsQ1nnxkK3uQgVUcltVnH+
VPb1OcoFtEHxENedcnKe+6N8m+r+50B6pHF/dJ4JJVASpXyeikLIHOI6Vd4W
RhFfn4GZyklci+zIhk4YiA4ogILIH+Iar+nr09yfKa7f2WO/Blncnwv4EJ+g
FVqjKco5DlM8L8MxOMQc1lpWnqcLcT0OCzEPFdtuIa535ZX2jb7ojfEhrgut
/VHug8ZSB0uZqAXeN5TnylutC+0VWtfaN3ZgJx5I4rzUcG60RG23o/kdhK7o
gV6Y4DotQsyxAY6b1ktzX2+GySGuoz642bHqF2Lu1XZslUt5HB/VuQVdQtx/
tG9oHWlPa+B7a7mf1Ty/inmK+9bbsVKcyzofdnouDoa4ZrXetS9t8bMU/w0e
o56bnVj0Rh98TflL5QjneZGCruiObj7XO2JdiHtCHcodUE3rEaVQE43RHH3x
EPXO4nGNm3IO5EJu7Tvoieqo4XZOOk+0BtWHLEiPjZ4LzYv21eUhvqd+D/G9
9jM2YZLnS2020fvBz0rxczMim9vMiczIkMS+6XrWJMZE/VQc9oX4XjuBwzii
vEvi/nnKudQeLdDIseqFHniEuo+FuHdpH3sQT+IPXERr6tRHJ3R2O4pnPbXL
/7NDXLMTPa5KXL8ON6EVajtuivkX/N+O441o5v60SeIc1UVb12+IWn5uAz9L
c6R9T/uS9m29y57FKZzGK8pr6vXznGbyvJTHyyHu89rzSztPSuIY5Y9CjGdH
P6sLCqOo5+U5/n/UzzoT4rtMedLS+dPUudTEsb3Mf3/ikudd3za/YS3WY3WI
c5oO+fAV5U/xcYj9nJnEedzKcS82J7HPBT2WYkls85LHe9Htr8TRENssm8Q+
pOH4DMcBSZzTMh5Xfq8jjf2DEMeruBXyHGl9fR5i3zYl8VzxLOc81D6md5ze
43pX7sMU56He17s5juO4P4nv0F8o/+r834/7Q9yHL3js6sOeEPc9xUHv0HXY
lcT9cBu2aH1hOzb4fJOva+wrQvyG0frajoqOg/pcIonx13rUt9Yy178+if3R
N6S+CTd6jrYiLf8lSayj/UrfZvrOLOC5UAy1TrU2tdb+4b9/8TeC773C6bYQ
v/H+8/n2/7WvHNAcVUCVJPZZ3136DtF3hfbAdc6Zyq5T1fHUs1KdD0U8dwV8
XsJ5e9brQuuxlHP+sufiivtZPIl9O+z80fen1oX2EO3P+vZODfEbZo/nTvNb
wfeqD1cB5RlOwA==
           "]], 
          Polygon3DBox[{{190, 188, 187, 189}, {182, 183, 185, 184}, {133, 183,
            182, 134}, {440, 423, 416, 417}, {187, 188, 367, 383}, {391, 388, 
           389, 392}, {36, 43, 42, 35}, {42, 43, 132, 131}, {416, 423, 424, 
           425}, {385, 388, 391, 386}}], 
          Polygon3DBox[{{383, 385, 386, 205, 202}, {438, 440, 417, 419, 
           439}, {184, 190, 189, 199, 206}, {367, 369, 384, 385, 383}, {389, 
           390, 393, 394, 392}, {185, 195, 196, 190, 184}, {132, 129, 128, 
           133, 131}, {392, 394, 438, 437, 435}, {131, 133, 134, 45, 44}, {
           394, 399, 422, 440, 438}, {29, 36, 35, 34, 27}}], 
          Polygon3DBox[CompressedData["
1:eJxF1nWUV0UYxvG59BJSsqSk1Ep3SCMl3UvKEhLKIp3KQQWR3qUbkVAWkZJj
HbG7AzHB7kSx9fucec7xjw97+d25c995Z96ZWyUjs//UXCGEuciHygn/4F/+
/IQfUSWJ93egLtcNUB/1sBWbsQ8HcQD7kSA3cuF3nvstxD63YjNWIxt7sDvE
Pusk8V1/8/+/0JDry1AWZVARFVAeWdxfj9Jc5/G7rkhin2tDvE48ln/4sw93
4jDuxRHco9+4vxe7sAOf89s5fBTi9Sf4DF/gvRDfVRz5UBu1PMYTOIlTSezz
AG334zgO4Tw+cAx7kYO73eag29yFP1DJYxyDa/Azv63DGqwKcexrnUPFUABp
KOdcVXXOa/laOVSfF0NsUwPVUB01ndvXuPcKXsbrftcGzT2qorr6Q03UwBZs
cpt1ntONIc6d1swP+B7f4c8Q33U5uqINmqMpOqM92uEqdEJHFEQqBmOIY05z
TrQG2qKDnx2NZ3jH03gKz+I0HsGvIeZwtN/VBd28lmo6hiZo7D5vpP0DfvZR
nMRNIcbWG61wJfqiu5/VWJqhJVqgEZbwzJN4DPfjFBaH+Gxr9McgDMQA9PGz
PdDPbVp5jI3dZx/HoFw9TF8P4jm8gOc99ku5VxLDMBxDncNB3NuFnZ67bRiA
FO7lR17nvIDX9ni32eia3Y7BntMLIa5J5fYXfIvhbqNaviPEvWI2itBXKZRI
YmypftdR12DhJMagmlItqzaPoVASn1Vshb0GNJZ09PJa0hr+0LV5CdfF/K5C
vi6KkRjlnCgG1fKnuJrrEW6jZxXznhD3Cu2B2ou012kP1N510bV5wWv7a3yF
L0OsnU3O7VGeOYYZmIvr+e06LOF6GY4nca9Q7Z9AU7TFGPRHP/TRmsUoDMEI
jMZIrUN0QQsU1bjRCc1RWvOPgsqr2/RCT/Tws501dnRDd1RESZRAR7QPscY7
oCEaoCWGYajXgOY63bEtxX14KcQaVO0sRCra+dkmaIb8Hku614bOlDmYpbpA
a0wIca0OdE6aOIb6zlVjNMI6cpiNTUk8g9ZgeRJzqGe1tsc75r5YjrHIcG7n
YaVzleLYCiE3CqAe6nos2hMWYT56oyzKOOddnNvKqIDyIe6ZlZzbop4L9ZkX
xVAkxL20tvtcgDbO+QK/S3vOFOekFUp5joo7tyme63Lo6rkr5TWg2G5zrrSW
xnnsy0I842/3tXKyApPdpp9zpbke5VxVQxrqOCeKWWeB9sbTeBcfh1gL32Bq
iGs+EzdgNXOyAhOxKom1oDYzMD3EM3c7tqFnEmtTZ67OyvdDPHNV49oTdHa/
jXdCPCt1tr6Bt0KstVlYoD0c0xzDHK7nYT4WYaHbzMZMTE9im1muWZ3FOrvP
hfgNoBjOYw/3dmOn9grHvMvvUp9HtHdhA7KSeAZpTz7rmM84V4e4dxg52gMc
m2LQHv5iiHtCjsd41rk94zG+GuIZrT2nodeM1oZqR3tIfbdRTvSNoW+eN8P/
Z7qu9c2jvVffTvqWWJnEOVqbxJpaipux3mPZiC1J/MZTralGVMuNvMa0lvg5
zPScaq7V50Sur/UaPum1oT1TZ5BqfBptJiexzSStFe2VSVwzmrssx5Dt2LRH
qab07dE+xL1FtTBFfSRxLSkG9TkpiX0qBo3lVtyCxZigNeg2imEqxmEsxvs6
A4+HeHZnuI1iy8RD/PZEiLWc22NXTeWBvqP/A8DgPGg=
           "]]}],
        Lighting->Automatic]}, {}, {}, {}, {}}, 
     {Thickness[Large], Line3DBox[CompressedData["
1:eJwt0DtLggEYBeCvcC1ci4ICtyBpCRpaUjJTI00HKzWMLlTUUkpBQ+GWrUEE
DoHQ/fYf+jm1NdnQIzQ8w3nhDOcdrR7k9nuCIFijlxA7XNDkmBIVjqgzrhBl
gl+5wzdf/DDnPssMMZIsskSeAjmyzDPFJNMkyFBkhWXKlFglRZobbmlxzx1t
rrnijVdOOeGBJ1545pFDanzwziXbbLHLHptssM6nbeec0WCMCMMMMUKyu58E
gwywQJoMKeLE6KeP8P/v/wDLDikg
       "]]}},
    VertexNormals->CompressedData["
1:eJztms0NwjAMhQOMwY/UGZiBBRgBiTNntmALVuiNXZgEzpWiJK3t57jv9KlS
m9jPL24adbg9rvdtSmncpLT783B+Xd7P7wfFY+Z6D44Lzbk6TPWsvb/0XK5O
aH28xCOls5VPpPyjHU9U5vxQq7+UzqeZ45TinuZXOz79Y1vHpfVFr5+oPtF6
X1jr7zV+LT3QcVjnic4b1QdK86J1iU76kKT+9jpIjR+1Xt76ErkOP3j/XmjV
KTqj5iuVl/Z5KaoOuflaz/2i+4jUofQ5YS/9xEt/4PoleyR9S2r6CR2Hl3h6
Yet/I972G9J19+ofr/qT66B1f106X+s+5weieBii
     "]], GraphicsComplex3DBox[CompressedData["
1:eJxllllIVGEYhsclHdxSR8euioQWKtLUjKj0FLhEojZ1UVFiVlomqEVdxIiO
WUIrJJKRoAaWCYEZdVEUWhRdWJmJSRthYkJo0uZMjZ7mYp7/wDcXg/jMO++3
/v85C4vKbQf8TSZTvecT4PlsOzaU4LZbtf684raKkxZtZeCHXqcepX0+VZfl
rja4a/vqeKfb4JNzb1SW1xh6fAo3BjnMgbFazqG6yY7qSK0p3V4xpccoLn3s
fllbg/189XCZDz7W+x9zO+yRWmjCyx1T7hhtMul6YWudVfnAJ5I2fNH1OO11
887U7tlw5Y8en3zX0Suj1Ralh6OXPlJP3KsPPgWZZ+JUnujRwYvTwqa7nAan
XulPnvSNPsPRq7l4Of7w0J9jiS0OQ48P8cNKOi9Gz4b41Ct9pF7yNvOe/Chn
iOrDpoJnvcP+0WqOcPaKvvF79NQLZ9+kHh/mRV2Zf8Lm2AIitKHBkXXjNVZV
F7yup6F8zWyc4q9WlHSne3zRkyf6X2vPHmwyxap64a2L1jd/rbIqjg968uku
Wzyi1RqcuHB84AXzgy+99fQLPfmgj2pJbsyuMfYP/v77ud9ZVRaVD/2Eo8df
ni+49KfeR2dGn1+eiVF64sLR0wd8now/dYzajX0mLlyeI/Odw4+HdWP+7IPk
+HDeOV9w+sn+48/5gst82Cv6T1w484JzftHjQz70gbhw5oIP/ZGcPPGhXuZL
XPzh5KnuE+85Qo8Pdck84cwXTr1yvsTlHJE/nHOn6vXmg57zgh5/5ggnH3Wf
e32knvxlP+k/Ppwv+s/9xp7Rf8nRy3tPnZtV/8aPzASr/yVP2X33TZ/n3iJP
voeTJ/3pKdv3ra/a0NPPqZx7S2v1cKWHo2de+KCHq+fQtYedw25jXpITV+ZD
/6WefNg39HD2k3rpJ3o4vsxXxfXqmSN1oYejlz5ST1zyx1/GZR9kXfiwb/jI
+2p4+S6Xw+UbF05/+J69Ck08f/qHyeyzT3DmDs/oilgy9dfYv/r+mjTbhO8+
DiQ39tl0g88rykxMdRp8cL/5dorb0HOOeF5IP/6+G7h1s3I6WNVFP9mfvT3t
J0r1EMXlPtNnqYcTB87/cs/xZ9+kHp7dsnlLqcvXB44P3/McdBy/0J43E+7z
/rDM8iIj3j9UcX4HJy68qCF3bIE7xOc+RC/jwuX7DPmo/fDe8zyv8ec+h+MD
Z77oiUuect7Elc9B/GU+vFfAicv7A/w/PJrb/g==
    "], {{
      {RGBColor[0.219993, 0.370006, 0.059999], Opacity[0.7], EdgeForm[None], 
       StyleBox[
        GraphicsGroup3DBox[{
          Polygon3DBox[{{2, 3, 1}, {32, 33, 31}, {2, 6, 27}, {35, 36, 34}, {8,
            6, 4}, {5, 7, 52}, {18, 14, 9}, {16, 17, 15}, {11, 53, 9}, {23, 
           13, 10}, {46, 44, 40}, {126, 121, 122}, {45, 132, 34}, {41, 47, 
           40}, {42, 44, 71}, {63, 130, 68}, {30, 29, 27}, {38, 39, 37}, {72, 
           73, 71}, {25, 24, 22}, {20, 21, 19}, {12, 14, 49}, {63, 64, 62}, {
           69, 70, 68}, {59, 54, 52}, {42, 3, 26}, {75, 61, 58}, {66, 67, 
           65}, {74, 7, 1}, {60, 51, 48}, {126, 131, 125}, {59, 67, 58}, {66, 
           64, 55}, {18, 54, 48}, {56, 57, 55}, {78, 77, 76}, {5, 53, 4}, {20,
            24, 76}, {74, 73, 65}, {8, 21, 15}, {11, 13, 19}, {129, 130, 
           128}, {129, 135, 133}, {46, 36, 37}, {123, 124, 122}, {95, 97, 
           102}, {94, 96, 98}, {84, 86, 103}, {91, 99, 90}, {50, 51, 101}, {
           117, 114, 112}, {35, 121, 31}, {83, 85, 88}, {56, 70, 88}, {60, 61,
            98}, {100, 96, 93}, {75, 57, 90}, {91, 85, 82}, {94, 99, 92}, {28,
            43, 26}, {84, 114, 82}, {83, 89, 81}, {32, 127, 134}, {72, 39, 
           62}, {105, 106, 104}, {38, 33, 128}, {113, 116, 112}, {117, 97, 
           92}, {69, 110, 87}, {108, 109, 107}, {111, 80, 79}, {113, 115, 
           120}, {118, 116, 119}}], 
          Polygon3DBox[{{77, 16, 15, 21, 20, 76}, {7, 74, 65, 67, 59, 52}, {
           80, 78, 76, 24, 25, 79}, {3, 42, 71, 73, 74, 1}, {54, 59, 58, 61, 
           60, 48}, {73, 72, 62, 64, 66, 65}, {114, 84, 103, 115, 113, 112}, {
           21, 8, 4, 53, 11, 19}, {64, 63, 68, 70, 56, 55}, {97, 117, 112, 
           116, 118, 102}, {24, 20, 19, 13, 23, 22}, {29, 28, 26, 3, 2, 27}, {
           33, 32, 134, 135, 129, 128}, {67, 66, 55, 57, 75, 58}, {47, 45, 34,
            36, 46, 40}, {44, 46, 37, 39, 72, 71}, {132, 123, 122, 121, 35, 
           34}, {124, 105, 104, 131, 126, 122}, {121, 126, 125, 127, 32, 
           31}, {70, 69, 87, 89, 83, 88}, {130, 129, 133, 110, 69, 68}, {43, 
           41, 40, 44, 42, 26}, {96, 94, 92, 97, 95, 93}, {99, 91, 82, 114, 
           117, 92}, {14, 18, 48, 51, 50, 49}, {61, 75, 90, 99, 94, 98}, {39, 
           38, 128, 130, 63, 62}, {57, 56, 88, 85, 91, 90}, {116, 113, 120, 
           109, 108, 119}, {6, 2, 1, 7, 5, 4}, {13, 11, 9, 14, 12, 10}, {53, 
           5, 52, 54, 18, 9}, {51, 60, 98, 96, 100, 101}, {17, 30, 27, 6, 8, 
           15}, {36, 35, 31, 33, 38, 37}, {85, 83, 81, 86, 84, 82}}]}],
        Lighting->Automatic]}, {}, {}, {}, {}}, 
     {Thickness[Large], 
      Line3DBox[{10, 12, 49, 50, 101, 100, 93, 95, 102, 118, 119, 108, 107, 
       109, 120, 115, 103, 86, 81, 89, 87, 110, 133, 135, 134, 127, 125, 131, 
       104, 106, 105, 124, 123, 132, 45, 47, 41, 43, 28, 29, 30, 17, 16, 77, 
       78, 80, 111, 79, 25, 22, 23, 10}]}},
    VertexNormals->CompressedData["
1:eJxTTMoPSmViYGBoB2JmIJYxnuC6vPLR/lF6lKYHLQulpQeJe0Zp6sSn7CBx
zyg9So/SQ49GLz9Gy5NRepQepWE0AICsK90=
     "]], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{0, 0, 0}, 0],
       "Ball"]},
     "1"],
    Annotation[#, 1, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{0, 0, 3}, 0],
       "Ball"]},
     "2"],
    Annotation[#, 2, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{0, 3, 0}, 0],
       "Ball"]},
     "3"],
    Annotation[#, 3, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{0, 3, 3}, 0],
       "Ball"]},
     "4"],
    Annotation[#, 4, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{3, 0, 0}, 0],
       "Ball"]},
     "5"],
    Annotation[#, 5, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{3, 0, 3}, 0],
       "Ball"]},
     "6"],
    Annotation[#, 6, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{3, 3, 0}, 0],
       "Ball"]},
     "7"],
    Annotation[#, 7, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[SphereBox[{3, 3, 3}, 0],
       "Ball"]},
     "8"],
    Annotation[#, 8, "Tooltip"]& ]},
  BaseStyle->{
    Specularity[
     GrayLevel[0.5], 100]},
  Boxed->False,
  Lighting->"Neutral",
  SphericalRegion->True]], "Output"]
}, Open  ]],

Cell["\<\
Several {1,1}-planes in a periodically repeated 2D square supercell with \
user-defined layout:\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CrystalPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"3", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "3"}], "}"}]}], "}"}], ",", 
   RowBox[{"Sysdim", "\[Rule]", "2"}], ",", 
   RowBox[{"CellLineStyle", "\[Rule]", "2"}], ",", 
   RowBox[{"LatticePlanes", "\[Rule]", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], ",", "dist"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"dist", ",", "1", ",", "5"}], "}"}]}], "]"}]}], ",", 
   RowBox[{"ContourStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<TerreVerte\>\"", ",", 
      RowBox[{"Thickness", "[", ".02", "]"}]}], "}"}]}]}], "]"}]], "Input",
 CellContext->"Global`"],

Cell[BoxData[
 GraphicsBox[{{
    {GrayLevel[0], Thickness[0.01], LineBox[{{0, 0}, {6, 0}}]}, 
    {GrayLevel[0], Thickness[0.01], LineBox[{{0, 3}, {6, 3}}]}, 
    {GrayLevel[0], Thickness[0.01], LineBox[{{0, 6}, {6, 6}}]}, 
    {GrayLevel[0], Thickness[0.01], LineBox[{{0, 0}, {0, 6}}]}, 
    {GrayLevel[0], Thickness[0.01], LineBox[{{3, 0}, {3, 6}}]}, 
    {GrayLevel[0], Thickness[0.01], LineBox[{{6, 0}, {6, 6}}]}}, {
    {GrayLevel[0], Thickness[0.01], Arrowheads[0.10500000000000001`], 
     ArrowBox[{{0, 0}, {3, 0}}]}, 
    {GrayLevel[0], Thickness[0.01], Arrowheads[0.10500000000000001`], 
     ArrowBox[{{0, 0}, {0, 3}}]}}, GraphicsComplexBox[CompressedData["
1:eJx1k01KA0EQhTtjNAFPEYgJeABXga5lJrjyCILLmLOYw0wED+DCrbsJgYAQ
QQhoXAjixEnVew0pmKaHmm+o6fp73bud3dxlIYSr+jlYW13h215XR7ZqF6Pq
iJ9H+bxf746UxbTeT3EM3jk+Xxgr5mfwf0m8AT+Ad2DzO5VLDfQaL9S28X8Z
++C14w34Xh1P5FoDraOmUWZifm+JB+ACPARbFzKx3rxHq7eF+B+Jl+AxeAW2
+oI86rnbaLaC32fiHFyAJ2CL84u6vpHPT7L8Tj+r5wvn/6VzChenKQ/mybxX
rq6lq9v3gX1i34aurwPXdz8Hzolz4xyb5kwdUBfUCXVDHVFX1FmTDqnTJh1T
59Q970Hl7gnvTThaXfkH0xyPGA==
    "], {{}, {}, 
     {RGBColor[0.219993, 0.370006, 0.059999], Thickness[0.02], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54}]}}], GraphicsComplexBox[CompressedData["
1:eJx1lc9LVFEUx59jPcdszPnhj23kDxCCVm7f2ZkytHDbLminaf9JmfSXVDoU
tGnX1pUTwYBgEAWVSBFp4znfzwMvvMcMdz537r3vnHPP+Z7bj7Y3HteyLHs6
/F6O8cwZv+bXdoefWeu/2Rp+3hb7l8PDWbvn8x8KH+b5/2PJJ+L74l/iLV84
o/2Hhf+9NqP9/ZK/iZ+JL8QxTNuGLxwUn3xi2sLO4+JIfDfhFfGFPx3L3bsv
Rezr6P1fSx6I74hPxOFPx967Hd+LJ76xbTH/Q/61rSveFD8Qh39t6/nCnxHP
fkv2n8q/li2JX4uXxeFPy176hrMibqml+/ot/5o2Kj4X5+J4T9N23K4/uq+m
/P8r/5r2OeGBOPybkn//5M+Uhd3nJffEz8XvxBGvW/bC5zM78PlJxW9E50za
uviVuCsOfxo24v7UtK6hfTWd09C5NVsV98Rh300LO0Yt8mdC8b+mOEzYgvhI
vCiOdTe077pt+4ZxCz9znTuu9+S2Kd4Th111i7iPWdRcXX6N6R7r5cg869jH
OXvJe3YTO7ALO7EbPxYTPxeSOBAX4kTciGMvifNBcg/cC/fEvXGP3eSe15M8
IC/IE/KGPCKvyDPyriovyVvymLyuynvqgjqhbqijPKkz6o46pC6pU+qWOqau
qXPqHh1AF9AJdAMdQVfQGXQHHUKX0Cl0Cx1D19A5dK9KF9FNdBRdrdJddBmd
RrfRcXQdnUf3q/oCfYM+Ql+hz9B3qvoSfYs+Rl+jz2VXnjn7D3ZmC/Y=
    "], {{}, {}, 
     {RGBColor[0.219993, 0.370006, 0.059999], Thickness[0.02], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 
       54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71,
        72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 
       89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103, 104, 
       105, 106, 107, 108, 109, 110, 111, 112, 113}]}}], 
   GraphicsComplexBox[CompressedData["
1:eJxdk0lOw0AQRS02iCnxEAe2CCc3YF87aEfZ5AiR2BFyFTA5SiblKDELTsLg
7v+y6Jas0iv3UMOv+/ly9nKWJMnj/9fZsO7s57db51Y3lWuqW3ttt4t2e2FO
/CZ+Fi/FwT+0kd94aZ/dbze0Y+deXFkj/hK/i7/F3rSlrbz/WvtKe/IHb6wV
1+KjeCoO8ZeyPRv7OAa29vf2rBLvxQ/igzjkM1A+fdnCPvzBVPkVyiNVXQrF
m8pfaF+qd3Ldm9lO7MQb8UQc8stVv8xCT3L1I1Ne+cniZx/nuGcSveOiOIiL
OImbPFZRnk1UB+pCnagbdTxEdd5HfaAv9Im+0cdp1Oc60gG6QCfoBh2hK3SG
7tAhukSn6BYdo2t0ju6ZA+aCOWFumKM/rbBOgA==
    "], {{}, {}, 
     {RGBColor[0.219993, 0.370006, 0.059999], Thickness[0.02], 
      LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 
       19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
        37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 
       53}]}}], {}, {}, 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[DiskBox[{0, 0}, 0],
       "Ball"]},
     "1"],
    Annotation[#, 1, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[DiskBox[{0, 3}, 0],
       "Ball"]},
     "2"],
    Annotation[#, 2, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[DiskBox[{3, 0}, 0],
       "Ball"]},
     "3"],
    Annotation[#, 3, "Tooltip"]& ], 
   TagBox[
    TooltipBox[
     {RGBColor[0.368417, 0.506779, 0.709798], 
      TagBox[DiskBox[{3, 3}, 0],
       "Ball"]},
     "4"],
    Annotation[#, 4, "Tooltip"]& ]}]], "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
Saveable->False,
WindowSize->{1664, 915},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
ShowSelection->True,
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "context" -> "Crystallica`", "keywords" -> {}, "index" -> True, "label" -> 
    "Crystallica Package Symbol", "language" -> "en", "paclet" -> 
    "Crystallica Package", "status" -> "", "summary" -> 
    "LatticePlanes is an option of CrystalPlot which gives a list of lattice \
planes to include in the plot.", "synonyms" -> {}, "title" -> "LatticePlanes",
     "type" -> "Symbol", "uri" -> "Crystallica/ref/LatticePlanes"}, 
  "SearchTextTranslated" -> "", "LinkTrails" -> ""},
TrackCellChangeTimes->False,
FrontEndVersion->"10.3 for Microsoft Windows (64-bit) (December 10, 2015)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "Info3576394248-7634371"->{
  Cell[760, 31, 208, 3, 40, "Print",
   CellTags->"Info3576394248-7634371"]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"Info3576394248-7634371", 62254, 1211}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[579, 22, 178, 7, 70, "Section"],
Cell[760, 31, 208, 3, 40, "Print",
 CellTags->"Info3576394248-7634371"],
Cell[971, 36, 172, 7, 29, "Item"],
Cell[1146, 45, 318, 6, 29, "Item"],
Cell[1467, 53, 253, 4, 29, "Item"],
Cell[1723, 59, 252, 5, 29, "Item"],
Cell[1978, 66, 205, 4, 29, "Item"],
Cell[2186, 72, 28, 0, 29, "Item"],
Cell[CellGroupData[{
Cell[2239, 76, 30, 0, 49, "Subsection"],
Cell[2272, 78, 33, 0, 30, "Text"],
Cell[2308, 80, 102, 2, 31, "Input"],
Cell[2413, 84, 52, 0, 30, "Text"],
Cell[CellGroupData[{
Cell[2490, 88, 553, 18, 31, "Input"],
Cell[3046, 108, 9504, 172, 375, "Output"]
}, Open  ]],
Cell[12565, 283, 117, 3, 30, "Text"],
Cell[CellGroupData[{
Cell[12707, 290, 1029, 29, 52, "Input"],
Cell[13739, 321, 42179, 737, 375, "Output"]
}, Open  ]],
Cell[55933, 1061, 119, 3, 30, "Text"],
Cell[CellGroupData[{
Cell[56077, 1068, 812, 24, 31, "Input"],
Cell[56892, 1094, 4320, 88, 407, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)
