(* ::Package:: *)

(*
Copyright (c) 2016 Bianca Eifert and Christian Heiliger, distributed under the MIT license
This file is part of Crystallica, do NOT load it separately.
Information about the Crystallica application can be found one level up from here.
*)
If[!TrueQ[$Context=="Crystallica`Private`"],Print["This package needs to be called from within Crystallica, aborting."];Abort[]];


(*default values for options: *)
Options[CrystalChange]={
Acell->0,
AddQ->False,
CenterAtOriginQ->False,
MoveOrigin->{0,0,0},
NewbasisQ->True,
RetractQ->False,
SortQ->False,
Sysdim->{1,1,1},
Reshape->IdentityMatrix[3]
};


CrystalChange::invalidstructure="This is not a valid crystal structure, something is wrong with the `1`.";
CrystalChange::atommismatch="Number of atoms is unclear: Received `1` atom coordinates, but `2` atom type specifications. Proceeding with `3` atoms.";
CrystalChange::molecule="CrystalChange cannot be applied to cartesian coordinates because all possible options refer to the lattice vectors.";
CrystalChange::reshape="The option value of Reshape contained non-integer values and will be rounded to `1`.";


CrystalChange[rawlattvec_List,coord_/;MatchQ[coord,{_List..}],conf_/;MatchQ[conf,{_Integer..}],rawoptions___]:=
Catch[Module[{options,x,retractoverride,lattvec,d2Q,sysdim,coordP,confP,zero,neworigin,reshape,retractq,retracttol,hexaQ,whichhex,vacuumfind,surfaceQ,addoverride,addtol,basis,atoms,latticeform,center},

(*debugging: *)
If[debug,Print["CrystalChange :: Entering function CrystalChange[rawlattvec,coord,conf,rawoptions] with the following input values:"]];
If[debug,Print["CrystalChange :: \trawlattvec = ",rawlattvec]];
If[debug,Print["CrystalChange :: \tcoord = ",coord]];
If[debug,Print["CrystalChange :: \tconf = ",conf]];
If[debug,Print["CrystalChange :: \trawoptions = ",Flatten[{rawoptions}]]];

(*parse options: *)
options=Select[Flatten[{rawoptions}],MatchQ[#,_Rule]&];

(*shortcut, because this will be used a lot: *)
x=(_?NumericQ);

(*some options cause a mandatory override of RetractQ or AddQ: *)
retractoverride=False;
addoverride=False;

(*check length of atom list vs. length of atom type list: *)
With[{l1=Length[coord],l2=Length[conf]},If[TrueQ[l1==l2],{coordP,confP}={coord,conf},
Message[CrystalChange::atommismatch,l1,l2,Min[l1,l2]];{coordP,confP}={coord[[;;Min[l1,l2]]],conf[[;;Min[l1,l2]]]}];];
confP=Round[Abs[confP]];
If[debug,Print["CrystalChange :: Atom coordinates after length matching safety net are ",coordP]];
If[debug,Print["CrystalChange :: Atom types after length matching safety net are ",confP]];

(*remove atoms of type 0: *)
zero=Flatten[Position[confP,0]];
coordP=coordP[[Complement[Range[Length[coordP]],zero]]];
confP=confP[[Complement[Range[Length[confP]],zero]]];
If[debug,Print["CrystalChange :: Atom coordinates after type-zero deletion are ",coordP]];
If[debug,Print["CrystalChange :: Atom types after type-zero deletion are ",confP]];

(*convert lattice vectors in different formats to standard format: *)
lattvec=Switch[rawlattvec,
{{x,x,x},{{x,x,x},{x,x,x},{x,x,x}}},rawlattvec[[1,#]]*rawlattvec[[2,#]]&/@Range[3],
{{x,x,x},{x,x,x}},Module[{a,b,c,\[Alpha],\[Beta],\[Gamma],veca,vecb,vecc,c2,c3},{{a,b,c},{\[Alpha],\[Beta],\[Gamma]}}={N[rawlattvec[[1]]],N[rawlattvec[[2]]]/180*\[Pi]};
veca=a*{1,0,0};vecb=b*{Cos[\[Gamma]],Sin[\[Gamma]],0};vecc=c*{Cos[\[Beta]],c2,c3};c2=(Cos[\[Alpha]]-Cos[\[Gamma]]*Cos[\[Beta]])/Sin[\[Gamma]];c3=Sqrt[1-(Cos[\[Beta]])^2-(c2)^2];Chop[{veca,vecb,vecc}]],
{{x,x},{{x,x},{x,x}}},rawlattvec[[1,#]]*rawlattvec[[2,#]]&/@Range[2],
{{x,x},{x}},Module[{a,b,\[Gamma]},{{a,b},{\[Gamma]}}={N[rawlattvec[[1]]],N[rawlattvec[[2]]]/180*\[Pi]};Chop[{a*{1,0},b*{Cos[\[Gamma]],Sin[\[Gamma]]}}]],
{{x,x,x},{x,x,x},{x,x,x}}|{{x,x},{x,x}},rawlattvec,
_,Message[CrystalChange::invalidstructure,"lattice vectors"];Throw[$Failed]];
If[debug,Print["CrystalChange :: Lattice vectors converted to standard notation ",lattvec]];

(*3D/2D: *)
d2Q=TrueQ[Length[lattvec]==2&&Length/@lattvec=={2,2}];
If[debug,Print["CrystalChange :: Check for 2D structure yielded ",d2Q]];

(*check atom coordinate syntax: *)
If[!MatchQ[coordP,If[d2Q,{{x,x}..},{{x,x,x}..}]],Message[CrystalChange::invalidstructure,"atom coordinates"];Throw[$Failed]];

(*system dimensions: *)
sysdim=Abs[Round[Sysdim/.options/.Options[CrystalChange]]];
sysdim=Switch[sysdim,{_Integer,_Integer,_Integer},sysdim,{_Integer,_Integer},Join[sysdim,{1}],{_Integer}|_Integer,Flatten[{sysdim,sysdim,sysdim}],_,{1,1,1}];
If[debug,Print["CrystalChange :: Option value for Sysdim resolved to ",sysdim]];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   MoveOrigin   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*change origin if desired: *)
neworigin=MoveOrigin/.options/.Options[CrystalChange]/.False|None->{0,0,0};
If[debug,Print["CrystalChange :: Option value for MoveOrigin resolved to ",neworigin]];
If[!TrueQ[neworigin=={0,0,0}],
(*retraction is necessary: *)
retractoverride=True;
neworigin=Which[
(*move origin onto an atom: *)
MatchQ[neworigin,_Integer]&&neworigin<=Length[confP],coordP[[neworigin]],
(*shift given in cartesian coordinates: *)
And@@(TrueQ[#<0]&/@neworigin),neworigin.Inverse[Transpose[lattvec]],
(*shift given in reduced coordinates: *)
True,neworigin];
If[debug,Print["CrystalChange :: Option value for MoveOrigin resolved to ",neworigin]];
(*move atoms to comply with the new origin: *)
coordP=#-Abs[neworigin]&/@coordP;
If[debug,Print["CrystalChange :: Option MoveOrigin changed atom coordinates to ",coordP]];
];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   Reshape   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*includes functionality of the option ConvCellQ which has been removed as of version 13*)
(*make supercells that are more complex linear combinations of the original lattice vectors: *)
reshape=Reshape/.options/.Options[CrystalChange];
(*allow input of known special cases with string identifiers: *)
reshape=reshape/.{"fcc"->{{1,1,-1},{1,-1,1},{-1,1,1}},"bcc"->{{1,1,0},{1,0,1},{0,1,1}}};
If[d2Q&&!MatchQ[reshape,{{x,x},{x,x}}],reshape=IdentityMatrix[2]];
If[!d2Q&&!MatchQ[reshape,{{x,x,x},{x,x,x},{x,x,x}}],reshape=IdentityMatrix[3]];
If[!TrueQ[reshape==Round[reshape]],Message[CrystalChange::reshape,Round[reshape]]];
reshape=Round[reshape];
If[debug,Print["CrystalChange :: Option value for Reshape resolved to ",reshape]];
If[!MemberQ[{IdentityMatrix[2],IdentityMatrix[3]},reshape],Module[{lattvecreshape,min,max},
(*retraction is necessary: *)
retractoverride=True;
(*new lattice vectors: *)
lattvecreshape=reshape.lattvec;
(*add atoms: *)
min=Min[reshape]-1;
max=Max[reshape]+1;
coordP=If[d2Q,Flatten[Table[(#+{a,b})&/@coordP,{a,min,max},{b,min,max}],2],Flatten[Table[(#+{a,b,c})&/@coordP,{a,min,max},{b,min,max},{c,min,max}],3]];
confP=Flatten[ConstantArray[confP,Length[coordP]/Length[confP]]];
(*project to new cell and change lattice vectors: *)
coordP=(coordP.lattvec).Inverse[lattvecreshape];
lattvec=lattvecreshape;
If[debug,Print["CrystalChange :: Option Reshape changed atom coordinates to ",lattvec]];
]];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   RetractQ   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*retraction, moving all atoms back into the cell: *)
If[debug,Print["CrystalChange :: Forced switch-ON for RetractQ option is ",retractoverride]];
retractq=RetractQ/.options/.Options[CrystalChange];
If[TrueQ[retractq==False]&&retractoverride,retractq=True];
(*RetractQ may already specify the retract tolerance, use default otherwise: *)
If[NumericQ[retractq],retracttol=N[retractq];retractq=True,retracttol=10^-4;retractq=TrueQ[retractq]];
If[debug,Print["CrystalChange :: Option value for RetractQ resolved to ",retractq]];
If[debug&&!TrueQ[retractq==False],Print["CrystalChange :: Retraction tolerance set to ",retracttol]];
If[retractq,Module[{new},
coordP=Partition[If[#>(1-retracttol),#-1,#]&/@Flatten[#-Floor[#]&/@coordP],If[d2Q,2,3]];
new=DeleteDuplicates[Transpose[{coordP,confP}],(Norm[Round[#1[[1]],retracttol]-Round[#2[[1]],retracttol]-Floor[Round[#1[[1]],retracttol]]+Floor[Round[#2[[1]],retracttol]]]<retracttol)&&(#1[[2]]==#2[[2]])&];
{coordP,confP}=Transpose[new]]];
If[debug&&!TrueQ[retractq==False],Print["CrystalChange :: Option RetractQ changed atom coordinates to ",coordP];Print["CrystalChange :: Option RetractQ changed atom types to ",confP]];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   Sysdim   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*periodic repetition of the cell: *)
coordP=If[d2Q,Flatten[Table[(#+{a,b})&/@coordP,{a,0,sysdim[[1]]-1},{b,0,sysdim[[2]]-1}],2],Flatten[Table[(#+{a,b,c})&/@coordP,{a,0,sysdim[[1]]-1},{b,0,sysdim[[2]]-1},{c,0,sysdim[[3]]-1}],3]];
confP=Flatten[ConstantArray[confP,Times@@(sysdim[[If[d2Q,{1,2},All]]])]];
If[debug,Print["CrystalChange :: Option Sysdim changed atom coordinates to ",coordP]];
If[debug,Print["CrystalChange :: Option Sysdim changed atom types to ",confP]];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   CenterAtOriginQ   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*center the cell at the origin if desired: *)
center=Flatten[{CenterAtOriginQ/.options/.Options[CrystalChange]}];
center=Switch[Length[center],1,ConstantArray[center[[1]],3],2,Join[center,{center[[2]]}],0,{False,False,False},_,center];
center=TrueQ/@center;
If[debug,Print["CrystalChange :: Option value for CenterAtOriginQ resolved to ",center]];
Table[If[center[[ii]],(coordP[[#,ii]]=If[TrueQ[coordP[[#,ii]]<=(sysdim[[ii]]-.5)],coordP[[#,ii]],coordP[[#,ii]]-sysdim[[ii]]];)&/@Range[Length[coordP]]],{ii,Range[If[d2Q,2,3]]}];
If[debug&&!TrueQ[center=={False,False,False}],Print["CrystalChange :: Option CenterAtOriginQ changed atom coordinates to ",coordP]];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   AddQ \[Rule] "hexa"   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*functionality of the option HexaAddQ which has been removed as of version 13*)
(*conventional hexagonal cells: *)
hexaQ=TrueQ[(AddQ/.options/.Options[CrystalChange])=="hexa"]&&(!d2Q);
If[hexaQ,whichhex=Switch[Round[VectorAngle@@lattvec[[#]]/Degree]&/@Subsets[Range[3],{2}],{90,90,120},"hexa",{90,120,90},"hexb",{120,90,90},"hexc",_,"none"]];
If[TrueQ[whichhex=="none"],hexaQ=False];
If[debug,Print["CrystalChange :: Option value AddQ->\"hexa\" resolved to ",hexaQ]];
If[hexaQ,
Module[{hexcoord,hexconf,new,hexaddtol=.01},
addoverride=True;
(*construct 2x2 supercell: *)
hexcoord=Switch[whichhex,
"hexa",Flatten[Table[(#+{a,b,c})&/@coordP,{a,0,0},{b,-1,0},{c,-1,0}],3],
"hexb",Flatten[Table[(#+{a,b,c})&/@coordP,{a,-1,0},{b,0,0},{c,-1,0}],3],
"hexc",Flatten[Table[(#+{a,b,c})&/@coordP,{a,-1,0},{b,-1,0},{c,0,0}],3]
];
hexconf=Flatten[Table[confP,{a,-1,0},{b,-1,0},{c,0,0}]];
(*add periodic duplicates: *)
new=Transpose[{hexcoord,hexconf}];
new=Join[new,#+{{1,0,0},0}&/@Select[new,#[[1,1]]<hexaddtol&],#+{{-1,0,0},0}&/@Select[new,#[[1,1]]>(1-hexaddtol)&]];
new=Join[new,#+{{0,1,0},0}&/@Select[new,#[[1,2]]<hexaddtol&],#+{{0,-1,0},0}&/@Select[new,#[[1,2]]>(1-hexaddtol)&]];
new=Join[new,#+{{0,0,1},0}&/@Select[new,#[[1,3]]<hexaddtol&],#+{{0,0,-1},0}&/@Select[new,#[[1,3]]>(1-hexaddtol)&]];
{hexcoord,hexconf}=Transpose[DeleteDuplicates[new,Abs[Norm[#1[[1]]-#2[[1]]]]<=.01&]];
(*cut off the 60 degree corners: *)
new=Transpose[{hexcoord,hexconf}];
Switch[whichhex,
"hexa",new=Select[new,(Sign[#[[1,2]]]==Sign[#[[1,3]]]||Abs[#[[1,2]]]+Abs[#[[1,3]]]<=1.01)&],
"hexb",new=Select[new,(Sign[#[[1,1]]]==Sign[#[[1,3]]]||Abs[#[[1,1]]]+Abs[#[[1,3]]]<=1.01)&],
"hexc",new=Select[new,(Sign[#[[1,1]]]==Sign[#[[1,2]]]||Abs[#[[1,1]]]+Abs[#[[1,2]]]<=1.01)&]
];
{coordP,confP}=Transpose[new];
If[debug,Print["CrystalChange :: Option value AddQ->\"hexa\" changed atom coordinates to ",coordP]];
If[debug,Print["CrystalChange :: Option value AddQ->\"hexa\" changed atom types to ",confP]];
]];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   AddQ \[Rule] "surfacecut"   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*functionality of the option SurfaceCutQ which has been removed as of version 13*)
(*surface cuts: *)
surfaceQ=TrueQ[(AddQ/.options/.Options[CrystalChange])=="surfacecut"]&&(!d2Q);
If[debug,Print["CrystalChange :: Option value AddQ->\"surfacecut\" resolved to ",surfaceQ]];
If[surfaceQ,vacuumfind=Module[{vacuum={},surfatoms,atomcycle,gap,direction},
For[direction=1,direction<=3,direction++,
surfatoms=Sort[#-Floor[#]&/@DeleteDuplicates[#[[direction]]&/@coordP]];
atomcycle=Partition[surfatoms,2,1,1,surfatoms[[1]]+1];
gap=Max[Abs[#[[2]]-#[[1]]]&/@atomcycle];
vacuum=AppendTo[vacuum,{direction,gap,gap*{1,1,1}.lattvec[[direction]],Select[atomcycle,Abs[#[[2]]-#[[1]]]==gap&]}];
];
vacuum[[Position[vacuum[[#,3]]&/@Range[3],Max[vacuum[[#,3]]&/@Range[3]]][[1,1]]]]
];
If[debug,Print["CrystalChange :: Option value AddQ->\"surfacecut\" resolved to ",surfaceQ]];
If[!(TrueQ[vacuumfind[[3]]>10]&&TrueQ[Length[vacuumfind[[4]]]==1]&&IntervalMemberQ[Interval[vacuumfind[[4,1]]],.5]),surfaceQ=False;If[debug,Print["CrystalChange :: Option value AddQ->\"surfacecut\" resolved to ",surfaceQ]]];
];
If[surfaceQ,Module[{surface,surfacefactor,surfaddtol=.02},
addoverride=True;
(*cut the surface: *)
surface=Select[Transpose[{coordP,confP}],#[[1,vacuumfind[[1]]]]<.5&];
surfacefactor=Ceiling[Max[#[[1,vacuumfind[[1]]]]&/@surface],.1];
(*add redundant atoms at all non-surface edges: *)
If[!TrueQ[vacuumfind[[1]]==1],surface=Join[surface,#+{{1,0,0},0}&/@Select[surface,Abs[#[[1,1]]]<surfaddtol&],#+{{-1,0,0},0}&/@Select[surface,Abs[#[[1,1]]]>(1-surfaddtol)&]]];
If[!TrueQ[vacuumfind[[1]]==2],surface=Join[surface,#+{{0,1,0},0}&/@Select[surface,Abs[#[[1,2]]]<surfaddtol&],#+{{0,-1,0},0}&/@Select[surface,Abs[#[[1,2]]]>(1-surfaddtol)&]]];
If[!TrueQ[vacuumfind[[1]]==3],surface=Join[surface,#+{{0,0,1},0}&/@Select[surface,Abs[#[[1,3]]]<surfaddtol&],#+{{0,0,-1},0}&/@Select[surface,Abs[#[[1,3]]]>(1-surfaddtol)&]]];
lattvec=#*Insert[{1,1},surfacefactor,vacuumfind[[1]]]&/@lattvec;
{coordP,confP}=Transpose[surface];
coordP=Round[#*Insert[{1,1},1/surfacefactor,vacuumfind[[1]]]&/@coordP,.00000001];
If[debug,Print["CrystalChange :: Option value AddQ->\"surfacecut\" changed atom coordinates to ",coordP]];
If[debug,Print["CrystalChange :: Option value AddQ->\"surfacecut\" changed atom types to ",confP]];
]];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   AddQ (regular cases)   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*add periodic duplicates of atoms at cell edges: *)
If[debug,Print["CrystalChange :: Forced switch-OFF for AddQ option is ",addoverride]];
If[!TrueQ[addoverride],
addtol=AddQ/.options/.Options[CrystalChange];
If[debug,Print["CrystalChange :: Option value for AddQ resolved to ",addtol]];
(*AddQ may already give the addition tolerance, and it may be anisotropic: *)
addtol=Switch[addtol,{_,_,_},addtol,{_,_},Join[addtol,{addtol[[2]]}],{_},ConstantArray[addtol[[1]],3],_,ConstantArray[addtol,3]];
addtol=(If[MatchQ[#,_?NumericQ],#,TrueQ[#]]&/@addtol)/.{True->.1};
addtol=If[NumericQ[addtol[[#]]]&&addtol[[#]]<0,-addtol[[#]]/Norm[lattvec[[#]]],addtol[[#]]]&/@Range[If[d2Q,2,3]];
If[debug,Print["CrystalChange :: Option value for AddQ resolved to addition tolerances ",addtol]];
Module[{new=Transpose[{coordP,confP}]},If[d2Q,
If[!TrueQ[addtol[[1]]==False],new=Join[new,#+{{sysdim[[1]],0},0}&/@Select[new,Abs[#[[1,1]]]<=addtol[[1]]-If[center[[1]],.5,0]&],#+{{-sysdim[[1]],0},0}&/@Select[new,Abs[#[[1,1]]]>=(sysdim[[1]]-addtol[[1]]-If[center[[1]],.5,0])&]]];
If[!TrueQ[addtol[[2]]==False],new=Join[new,#+{{0,sysdim[[2]]},0}&/@Select[new,Abs[#[[1,2]]]<=addtol[[2]]-If[center[[2]],.5,0]&],#+{{0,-sysdim[[2]]},0}&/@Select[new,Abs[#[[1,2]]]>=(sysdim[[2]]-addtol[[2]]-If[center[[2]],.5,0])&]]];,
If[!TrueQ[addtol[[1]]==False],new=Join[new,#+{{sysdim[[1]],0,0},0}&/@Select[new,Abs[#[[1,1]]]<=addtol[[1]]-If[center[[1]],.5,0]&],#+{{-sysdim[[1]],0,0},0}&/@Select[new,Abs[#[[1,1]]]>=(sysdim[[1]]-addtol[[1]]-If[center[[1]],.5,0])&]]];
If[!TrueQ[addtol[[2]]==False],new=Join[new,#+{{0,sysdim[[2]],0},0}&/@Select[new,Abs[#[[1,2]]]<=addtol[[2]]-If[center[[2]],.5,0]&],#+{{0,-sysdim[[2]],0},0}&/@Select[new,Abs[#[[1,2]]]>=(sysdim[[2]]-addtol[[2]]-If[center[[2]],.5,0])&]]];
If[!TrueQ[addtol[[3]]==False],new=Join[new,#+{{0,0,sysdim[[3]]},0}&/@Select[new,Abs[#[[1,3]]]<=addtol[[3]]-If[center[[3]],.5,0]&],#+{{0,0,-sysdim[[3]]},0}&/@Select[new,Abs[#[[1,3]]]>=(sysdim[[3]]-addtol[[3]]-If[center[[3]],.5,0])&]]];
];{coordP,confP}=Transpose[new]];
If[debug&&!MatchQ[addtol,{False..}],Print["CrystalChange :: Option AddQ changed atom coordinates to ",coordP]];
If[debug&&!MatchQ[addtol,{False..}],Print["CrystalChange :: Option AddQ changed atom types to ",confP]];
];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   NewBasisQ   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*project atoms into new basis: *)
With[{newbasis=TrueQ[NewbasisQ/.options/.Options[CrystalChange]]},
If[debug,Print["CrystalChange :: Option value for NewbasisQ resolved to ",newbasis]];
If[newbasis,
basis=(lattvec[[#]]*sysdim[[#]])&/@Range[If[d2Q,2,3]];atoms=(coordP.lattvec).Inverse[basis];If[debug,Print["CrystalChange :: Option NewbasisQ changed lattice vectors to ",basis]];If[debug,Print["CrystalChange :: Option NewbasisQ changed atom coordinates to ",atoms]],
basis=lattvec;atoms=coordP];
];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   Acell   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*lattice vector output format: *)
With[{acell=Acell/.options/.Options[CrystalChange]/.{True->1,Except[1|2]->0}},
If[debug,Print["CrystalChange :: Option value for Acell resolved to ",acell]];
latticeform=Switch[acell,
1,{N[Norm[basis[[#]]]]&/@Range[If[d2Q,2,3]],N[Normalize/@basis]},
2,{N[Norm[basis[[#]]]]&/@Range[If[d2Q,2,3]],If[d2Q,{N[VectorAngle@@basis]},N[VectorAngle@@basis[[RotateLeft[Range[3],#][[;;2]]]]&/@Range[3]]]/Pi*180},
_,basis];
If[debug&&MemberQ[{1,2},acell],Print["CrystalChange :: Option Acell changed lattice vectors to ",latticeform]];
];

(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)
(*   SortQ   *)
(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)

(*sort atoms by type: *)
With[{sort=SortQ/.options/.Options[CrystalChange]/.{Except[True|"Weak"|"weak"]->False}},
If[debug,Print["CrystalChange :: Option value for SortQ resolved to ",sort]];
If[MemberQ[{"Weak","weak"},sort],With[{sorted=Flatten[GatherBy[Transpose[{confP,atoms}],#[[1]]&],1]},{confP,atoms}=Transpose[sorted]]];
If[TrueQ[sort],With[{sorted=Sort[Transpose[{confP,atoms}]]},{confP,atoms}=Transpose[sorted]]];
If[debug&&MemberQ[{"Weak","weak",True},sort],Print["CrystalChange :: Option SortQ changed atom coordinates to ",atoms];Print["CrystalChange :: Option SortQ changed atom types to ",confP]];
];

If[debug,Print["CrystalChange :: Exiting function CrystalChange and returning to calling function or to user."]];
{latticeform,atoms,confP}]];


(*molecules cannot be supported;
note that this exception case does not cause problems even though CrystalPlot always calls CrystalChange (molecule or not), because
(a) CrystalPlot always calls with lattice vectors and therefore never ends up in this case anyway, and
(b) CrystalPlot explicitly resets lattice-related options if given, therefore it doesn't end up with incorrect physics despite the structure looking like a crystal to CrystalChange;
in the molecular case, CrystalPlot really only calls CrystalChange to benefit from things like syntax check and atom-type-zero deletion*)
CrystalChange[coord_/;MatchQ[coord,{_List..}],conf_/;MatchQ[conf,{_Integer..}],options___]:=(
If[debug,Print["CrystalChange :: Entering function CrystalChange[coord,conf,options] with the following input values:"]];
If[debug,Print["CrystalChange :: \tcoord = ",coord]];
If[debug,Print["CrystalChange :: \tconf = ",conf]];
If[debug,Print["CrystalChange :: \toptions = ",Flatten[{options}]]];
Message[CrystalChange::molecule];$Failed);


(*lattice vectors (cell without atoms) can also be handled, but require a dummy atom: *)
CrystalChange[lattvec_List,options___]:=Module[{structure},
If[debug,Print["CrystalChange :: Entering function CrystalChange[lattvec,options] with the following input values:"]];
If[debug,Print["CrystalChange :: \tlattvec = ",lattvec]];
If[debug,Print["CrystalChange :: \toptions = ",Flatten[{options}]]];
If[debug,Print["CrystalChange :: Calling new instance of function CrystalChange with different input."]];
structure=CrystalChange[lattvec,{If[MatchQ[lattvec,{{_,_},{_,_}}|{{_,_},{{_,_},{_,_}}}|{{_,_},{_}}],{0,0},{0,0,0}]},{1},
FilterRules[Flatten[{options}],{Sysdim,Reshape,Acell,NewbasisQ}]][[1]];
If[debug,Print["CrystalChange :: Exiting function CrystalChange and returning to calling function or to user."]];
structure
];
