(* Wolfram Language Package *)

(* Created by the Wolfram Workbench 22 oct. 2021 *)

(* ::Title:: *)
(*Equations of State of the Dynamic Viscosity and Self-Diffusion Coefficient for Potassium fluid*)

(* Title: EOSPotassium.m *)
(* Context: *)
(* 
  Authors:Fr\[EAcute]d\[EAcute]ric Aitken and Ferdinand Volino *)
(* Date: 22.10.2021, Grenoble *)

(* 
  Mathematica Version: 12.3 *)


(*******************************************)
BeginPackage["EOSPotassium`"]
(*******************************************)
Off[Protect::locked];
Off[Attributes::locked];
Off[ClearAll::wrsym];
Off[SetDelayed::write]
Off[SetAttributes::sym]
Unprotect@@Names["EOSPotassium`*"];
ClearAll@@Names["EOSPotassium`*"];

EOSPotassiumFunctions::usage = "List of public functions and parameters";

TmpK::usage = "Potassium fusion point temperature at atmopsheric pressure (K)";
TcK::usage = "Potassium critical point temperature (K)";
rhocK::usage = "Potassium critical point density (g/cm^3)";
PcK::usage = "Potassium critical point pressure (bar)";
MK::usage = "Potassium molar mass (g/mole)";
K0K::usage = "Characteristic elastic constant value (dyne/cm^2)";
nBK::usage = "Number of molecules in the basic unit cell";

Rho1atm::usage = "Density of potassium along the atmospheric isobar in the liquid phase (g/cm^3)";
RhoSatLiqK::usage = "Liquid Density of potassium along the coexistence curve (g/cm^3)";
RhoSatVapK::usage = "Vapor Density of potassium along the coexistence curve (g/cm^3)";
PsatK::usage = "Pressure along the coexistence curve of potassium (bar)";
LvK::usage = "Latent heat of vaporisation of potassium (kJ/moles)";
zK::usage = "Compressibility factor of gaseous potassium";

K0star1atm::usage = "Reduced elastic constant value along the atmospheric isobar in the liquid phase";
fN1atm::usage="Reduced fluctuative distance dN value along the atmospheric isobar in the liquid phase";
RhoKnu1atm::usage="Reduced density of the released gas along the atmospheric isobar in the liquid phase";
fqc1atm::usage="Non-dimensional function which represents the variations of the cut-off wave-vector modulus along the atmospheric isobar in the liquid phase";

RhoKnuVap::usage="Reduced density of the released gas in the gaseous phase only";
fqc::usage="Non-dimensional function which represents the variations of the cut-off wave-vector modulus in the gaseous phase only";

qc0K::usage="Characteristic cut-off wave-vector modulus (cm^-1)";
TtransVapK::usage="Transition temperature for gaseous potassium (K)";
NdNtcVapK::usage="Reduced coherence length for gaseous potassium";

etalKVap::usage="Potassium viscosity liquid term in the gaseous phase only (cP)";
etalK1atm::usage="Potassium viscosity liquid term along the atmospheric isobar in the liquid phase (cP)";
etaKnuKVap::usage="Potassium viscosity Knudsen term in the gaseous phase only (cP)";
etaKnuK1atm::usage="Potassium viscosity Knudsen term along the atmospheric isobar in the liquid phase (cP)";
etaKVap::usage="Potassium viscosity equation of state for gaseous phase only (cP)";
etaK1atm::usage="Potassium viscosity equation of state along the atmospheric isobar in the liquid phase (cP)";

DtK1atm::usage="Potassium self-diffusion coefficient equation of state along the atmospheric isobar in the liquid phase (cm^2/s)";
DtKVap::usage="Potassium self-diffusion coefficient equation of state for the gaseous phase only (cm^2/s)";

u2lVap::usage="Potassium Full mean square displacement <u^2> for the gaseous phase only (cm^2)";

(*CoefK::usage="A constant that multitplies the elastic parameter K*";
Coefd::usage="A constant that multitplies the dissipative distance d";
CoefN::usage="A constant that multitplies the distance parameter dN";*)


(*******************************)
Begin["`Private`"]
(*******************************)
Off[General::unfl];
Off[NIntegrate::inumr];
Off[General::stop];
Off[General::munfl];


(****************  EoSPotassiumFunctions **********************************)
SetAttributes[EOSPotassiumFunctions,HoldAll];
EOSPotassiumFunctions[a_String]:=ToExpression[StringJoin["?EOSPotassium`",a]];
EOSPotassiumFunctions[a_]:=ToExpression[
  StringJoin["?EOSPotassium`",StringDrop[StringDrop[ToString[Hold[a]],5],-1],"*"]
			];
EOSPotassiumFunctions[]:=ToExpression["?EOSPotassium`*"];

(* ::Input::Initialization:: *)
(*------------------------------------------------------------*)
SetAttributes[{MK,TcK,rhocK,PcK,TmpK,K0K,d,nBK},Constant];
(*Potassium fusion point temperature at atmopsheric pressure (K)*)
TmpK=336.65;(*K*)

(*Potassium critical point characteristics*)
TcK=2223;(*K*)
rhocK=0.194;(*g/cm^3*)
PcK=162.12;(*bar*)

MK=39.0983;(*g/mole, Molar mass of Water*)

(*Potassium constants of the model*)
K0K=5.4309*^10;(*dyne/cm^2*)
nBK=2;(*Number of molecules in the basic unit cell*)
d=0.01;(*Characteristic dissipative distance in cm*)

(* ::Input::Initialization:: *)
(*------------------------------------------------------------*)
(*Density of potassium along the atmospheric isobar in g/cm^3;if units are not precised then T=temperature in Kelvin and T \[LessEqual] 1030.793 K*)
Attributes[Rho1atm]={ReadProtected,NumericFunction};
Rho1atm[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &)] := 
  With[{Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"],T]}, With[{Rho = 0.93502 - 0.00038621*Temp + 2.6342*^-7*Temp^2 - 1.4093*^-10*Temp^3}, 
    If[Temp <= 1030.793, 
     If[QuantityQ[T], Quantity[Rho, ("Grams")/("Centimeters")^3],Rho]]]];

(*Liquid Density of potassium along the coexistence curve in g/cm^3*)
Attributes[RhoSatLiqK]={ReadProtected,NumericFunction};
RhoSatLiqK[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &)] := 
  With[{Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"],T]},
   With[{Tred = 1 - Temp/TcK}, 
    With[{Rho = rhocK*(1 + 0.67096*Tred^6.1442 + 3.3663*Tred^0.54393)}, 
     If[Temp <= TcK, 
      If[QuantityQ[T], Quantity[Rho, ("Grams")/("Centimeters")^3],Rho]]]]];

(*Vapor Density of potassium along the coexistence curve in g/cm^3*)
Attributes[RhoSatVapK]={ReadProtected,NumericFunction};
RhoSatVapK[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &)] := 
  With[{Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"],T]},
   With[{Tred = 1 - Temp/TcK}, 
    With[{logRho = -85.2049*Tred^9.74181 - 1.74903*Tred^(0.388134)-0.140485*Tred^(4/3) - 7.06168*Tred^(9/3)+97.3286*Tred*Exp[-Abs[(Temp+502.412)/598.176]^1.9]}, 
     If[Temp <= TcK, 
      If[QuantityQ[T], 
       Quantity[rhocK*10^logRho, ("Grams")/("Centimeters")^3],rhocK*10^logRho]]]]];

(*Pressure along the coexistence curve of potassium in bar between 298.15 K and Tc*)
Attributes[PsatK]={ReadProtected,NumericFunction};
PsatK[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &)] := 
  With[{Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"],T]},
   With[{exp = -(-0.097263 + 44.002) - 0.097263/(Temp/TcK)^1.9396+44.002*(Temp/TcK) - 42.095*(Temp/TcK)^0.51788*Log[Temp/TcK]}, 
    If[Temp <= TcK, 
     If[QuantityQ[T], Quantity[PcK*10^exp, "Bars"], PcK*10^exp]]]];

(*Derivative of the saturated vapor pressure function*)
Attributes[dPsatKdT]={ReadProtected,NumericFunction};
dPsatKdT[T_] := 10^(-43.904737000000004 - 0.097263/(T/TcK)^1.9396 + (44.002 T)/TcK - 42.095*(T/TcK)^0.51788 Log[T/TcK]) PcK Log[10] (-((42.095 (T/TcK)^0.51788)/T) + 44.002/TcK + 0.1886513148/((T/TcK)^2.9396 TcK) - (21.8001586 Log[T/TcK])/((T/TcK)^0.48212 TcK));(*bar/K*)

(*Latent heat of vaporisation in kJ/moles between TmpK and Tc*)
Attributes[LvK]={ReadProtected,NumericFunction};
LvK[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &)] := 
  With[{Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"], T]}, With[{ChaleurLatente = Temp*MK*(1/RhoSatVapK[Temp] - 1/RhoSatLiqK[Temp])*dPsatKdT[Temp]/10000},
    If[TmpK <= Temp <= TcK, If[QuantityQ[T], Quantity[ChaleurLatente, "Kilojoules"/"Moles"], ChaleurLatente]]]];

(*Compressibility factor of gaseous potassium; T in Kelvin and V in cm^3/mole*)
Attributes[zK]={ReadProtected,NumericFunction};
zK[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &), 
   V_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,("Centimeters")^3/("Moles")])) &)] := 
  With[{Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"],T], 
  	Vmol = If[QuantityQ[V], N@QuantityMagnitude@UnitConvert[V, ("Centimeters")^3/("Moles")],V]}, 
       With[{logB = (-2.01657 + 2982.8/Temp + Log10[Temp]), 
     logC = (2.66215 + 5710.87/Temp), 
     logD = (3.42866 + 9196.84/Temp)}, 
    1 - 10^logB/Vmol + 10^logC/(Vmol)^2 - 10^logD/(Vmol)^3]];

(* ::Input::Initialization:: *)
SetAttributes[{K0star1atm,fN1atm},Constant];

(*K0^*(\[Rho]) function ; rho=density in g/cm^3*)
K0star1atm = 0.225;
K0starlim[rho_] := 0.0001015*(rho/rhocK)^3;

(* ::Input::Initialization:: *)
(*Subscript[f, N](\[Rho]) function ; rho=density in g/cm^3*)
fN1atm = 158.45;
fNlim[rho_] := 6.563*(rho/rhocK)^2;

(* ::Input::Initialization:: *)
(*Subscript[Overscript[\[Rho], ~], Knu,0] funtion ; rho=density in g/cm^3*)
GamKnu[rho_] := 1 + (7 - 1)*Erf[(rho/0.01519)^1.5];
RhoKnuVap[T_,rho_] := (0.063039 + (0.025174 - 0.063039)*Erf[(rho/0.010419)^1.08])*Exp[-(1050/T)^GamKnu[rho]];
(*Chiong parameters:*)
RhoKnu1atm[T_] := ((1.318 - 0.262)*Exp[-Abs[(T - 336.65)/148.192]^0.86719] + 0.262);

(* ::Input::Initialization:: *)
(*fqc function;T=temperature in Kelvin; rho=density in g/cm^3*)
fqc1atm[T_] := (1 + (nBK^(1/3) - 1)*((T - 336.65)/407.187)^2/(1 + ((T - 636.78)/407.187)^2));

\[Alpha]qcK[T_] := 0.0437324*Exp[-(T/823.188)^6] + 0.0171575*Exp[-(T/1181.5)^10] + 0.05266266*Exp[-(T/2125.25)^22];
fqc[T_, rho_] := 1 + \[Alpha]qcK[T]*Exp[-(rho/RhoSatLiqK[T])] (1 - rho/RhoSatLiqK[T]) (rhocK/rho)^(4/3);

(* ::Input::Initialization:: *)
(*--- r=T/Tt (droite signifie r>1, soit haute temp\[EAcute]rature) -----------*)
(* Right Tt: r=T/Tt > 1, i.e. T>Tt *)
vd[r_] := \!\(\*
TagBox[
RowBox[{"1", "+", 
SuperscriptBox[
RowBox[{"(", 
RowBox[{"1", "-", 
RowBox[{"1", "/", "r"}]}], ")"}], 
FractionBox["1", "4"]]}],
CheckAbort[#, 
Defer[#]]& ]\);
Hnd[r_, N_] := (N^(vd[r] - 1) - 1)/(vd[r] - 1);
(* Left Tt: r=T/Tt, i.e. T<Tt*)
vg[r_] := 1 - Sqrt[1 - r];
Hng[r_, N_] := (N^(-1 + vg[r]) - 1)/(vg[r] - 1);

(* ::Input::Initialization:: *)
(*Cut-off wave-vector modulus in cm^-1; if units are not precised then \[Rho]=density in g/cm^3*)
qc0K[\[Rho]_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1, ("Grams")/("Centimeters")^3])) &)] := 
  With[{Na = 6.022136736*^23, Dens = If[QuantityQ[\[Rho]], N@QuantityMagnitude@UnitConvert[\[Rho], ("Grams")/("Centimeters")^3], \[Rho]]}, 
   With[{cutOff = (6 \[Pi]^2 (Na Dens)/(MK*nBK))^(1/3)}, 
    If[QuantityQ[\[Rho]], Quantity[cutOff, 1/("Centimeters")], 
     cutOff]]];

(*Transition temperature Subscript[T, t] in Kelvin; if units are not precised then \[Rho]=density in g/cm^3*)
Options[TtransVapK] = {CoefK -> 1};
TtransVapK[\[Rho]_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1, ("Grams")/("Centimeters")^3])) &), OptionsPattern[TtransVapK]] :=
   With[{Dens = If[QuantityQ[\[Rho]], N@QuantityMagnitude@UnitConvert[\[Rho], ("Grams")/("Centimeters")^3], \[Rho]]}, 
   With[{Tt = OptionValue[CoefK]*K0starlim[Dens]*TcK rhocK/Dens}, 
    If[QuantityQ[\[Rho]], Quantity[Tt, "Kelvins"], Tt]]];

(*Reduced cohenrence length N; if units are not precised then \[Rho]=density in g/cm^3*)
Options[NdNtcVapK] = {CoefN -> 1, Coefd -> 1};
NdNtcVapK[\[Rho]_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1, ("Grams")/("Centimeters")^3])) &), OptionsPattern[NdNtcVapK]] := 
 With[{Dens = If[QuantityQ[\[Rho]], N@QuantityMagnitude@UnitConvert[\[Rho], ("Grams")/("Centimeters")^3], \[Rho]]}, 
  1 + (OptionValue[Coefd]*d*qc0K[rhocK])/(2 \[Pi]) OptionValue[CoefN]*fNlim[Dens]];(* d is in cm *)

(* ::Input::Initialization:: *)
(*Viscosity liquid term with units written in etaTlDefaultUnit; if units are not precised then \[Rho]=density in g/cm^3 and T=temperature in Kelvin*)
etalKVap[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &), \[Rho]_?(NumberQ[#] || (QuantityQ[#] &&(CompatibleUnitQ[#1, ("Grams")/("Centimeters")^3])) &), 
   opts : OptionsPattern[{TtransVapK, NdNtcVapK, etaKVap}]] := 
  With[{Temp =If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"],T], Dens =If[QuantityQ[\[Rho]],N@QuantityMagnitude@UnitConvert[\[Rho], ("Grams")/("Centimeters")^3], \[Rho]]}, 
   With[{ViscoLiq = Quantity[(100 OptionValue[Coefd]*d Sqrt[OptionValue[CoefK]*K0starlim[Dens]*K0K*Dens])/Hnd[Temp/TtransVapK[Dens, FilterRules[{opts}, Options[TtransVapK]]],NdNtcVapK[Dens, FilterRules[{opts}, Options[NdNtcVapK]]]], 
       "Centipoise"]}, If[CompatibleUnitQ[OptionValue[etaKVapDefaultUnit], "Centipoise"], If[QuantityQ[T] || QuantityQ[\[Rho]], UnitConvert[ViscoLiq, OptionValue[etaKVapDefaultUnit]],QuantityMagnitude@UnitConvert[ViscoLiq, OptionValue[etaKVapDefaultUnit]]],Print["Units pb"]]]];

etalK1atm[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &),opts : OptionsPattern[{TtransVapK, NdNtcVapK, etaK1atm}]] := 
  With[{Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"],T]}, With[{ViscoLiq = Quantity[(100 *OptionValue[Coefd]*d \[Sqrt](OptionValue[CoefK]*K0star1atm*K0K*Rho1atm[Temp]))/
        Hnd[Temp/(OptionValue[CoefK]*K0star1atm*TcK rhocK/Rho1atm[Temp]),1 + (OptionValue[Coefd]*d *qc0K[rhocK])/(2 \[Pi])OptionValue[CoefN]*fN1atm], "Centipoise"]}, 
    If[CompatibleUnitQ[OptionValue[etaK1atmDefaultUnit], "Centipoise"],If[QuantityQ[T], UnitConvert[ViscoLiq, OptionValue[etaK1atmDefaultUnit]],QuantityMagnitude@UnitConvert[ViscoLiq, OptionValue[etaK1atmDefaultUnit]]],Print["Units pb"]]]];

(*Viscosity Knudsen term with units written in etaTlDefaultUnit; if units are not precised then \[Rho]=density in g/cm^3 and T=temperature in Kelvin*)
etaKnuKVap[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &), \[Rho]_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1, ("Grams")/("Centimeters")^3])) &), 
   opts : OptionsPattern[etaKVap]] := 
  With[{R = 8.3145192448*^7, Temp = If[QuantityQ[T],N@QuantityMagnitude@UnitConvert[T, "Kelvins"], T],Dens = If[QuantityQ[\[Rho]],N@QuantityMagnitude@UnitConvert[\[Rho], ("Grams")/("Centimeters")^3], \[Rho]]}, 
   With[{ViscoKnu = Quantity[100*RhoKnuVap[Temp, Dens]*((R*Temp/MK)^0.5)*2*\[Pi]/qc0K[rhocK],"Centipoise"]}, 
    If[CompatibleUnitQ[OptionValue[etaKVapDefaultUnit], "Centipoise"], If[QuantityQ[T] || QuantityQ[\[Rho]], UnitConvert[ViscoKnu, OptionValue[etaKVapDefaultUnit]], 
      QuantityMagnitude@UnitConvert[ViscoKnu, OptionValue[etaKVapDefaultUnit]]],Print["Units pb"]]]];
     
(*Parametres de Chiong*)
etaKnuK1atm[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &), opts : OptionsPattern[etaK1atm]] := 
  With[{R = 8.3145192448*^7,Temp = If[QuantityQ[T],N@QuantityMagnitude@UnitConvert[T, "Kelvins"], T]}, 
   With[{ViscoKnu = Quantity[100*RhoKnu1atm[Temp]*((R*Temp/MK)^0.5)*2*\[Pi]/qc0K[rhocK],"Centipoise"]}, 
    If[CompatibleUnitQ[OptionValue[etaK1atmDefaultUnit], "Centipoise"], If[QuantityQ[T],UnitConvert[ViscoKnu, OptionValue[etaK1atmDefaultUnit]], 
      QuantityMagnitude@UnitConvert[ViscoKnu, OptionValue[etaK1atmDefaultUnit]]],Print["Units pb"]]]];

(*Potassium viscosity EOS with units written in etaTlDefaultUnit; if units are not precised then \[Rho]=density in g/cm^3 and T=temperature in Kelvin*)
Options[etaKVap] = {etaKVapDefaultUnit -> QuantityUnit[Quantity["Centipoise"]]};
etaKVap[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &), \[Rho]_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1, ("Grams")/("Centimeters")^3])) &),opts : OptionsPattern[{TtransVapK, NdNtcVapK, etaKVap}]] := etalKVap[T, \[Rho], opts] + etaKnuKVap[T, \[Rho], FilterRules[{opts}, Options[etaKVap]]];

Options[etaK1atm] = {etaK1atmDefaultUnit ->QuantityUnit[Quantity["Centipoise"]]};
etaK1atm[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &),opts : OptionsPattern[{TtransVapK, NdNtcVapK, etaKVap}]] := etalK1atm[T, opts] + etaKnuK1atm[T, FilterRules[{opts}, Options[etaK1atm]]];

(* ::Input::Initialization:: *)
(*Potassium self-diffusion coefficient with units written in DtKDefaultUnit; if units are not precised then \[Rho]=density in g/cm^3 and T=temperature in Kelvin*)
Options[DtK1atm] = {DtK1atmDefaultUnit -> QuantityUnit[Quantity[("Centimeters")^2/("Seconds")]]};
DtK1atm[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &),opts : OptionsPattern[{TtransVapK, NdNtcVapK, DtK1atm}]] := 
  With[{kB = 1.38065812*^-16, Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"], T]}, 
   With[{CoefDiff = Quantity[(Hnd[Temp/(OptionValue[CoefK]*K0star1atm*TcK rhocK/Rho1atm[Temp]),1 + (OptionValue[Coefd]*d *qc0K[rhocK])/(2 \[Pi]) OptionValue[CoefN]*fN1atm]*fqc1atm[Temp]*qc0K[Rho1atm[Temp]]*kB*Temp*1.5)/(\[Pi]^2 OptionValue[Coefd]*d Sqrt[OptionValue[CoefK]*K0star1atm*K0K*Rho1atm[Temp]]), ("Centimeters")^2/("Seconds")]}, 
    If[CompatibleUnitQ[OptionValue[DtK1atmDefaultUnit], ("Centimeters")^2/("Seconds")], 
     If[QuantityQ[T],UnitConvert[CoefDiff, OptionValue[DtK1atmDefaultUnit]],QuantityMagnitude@UnitConvert[CoefDiff, OptionValue[DtK1atmDefaultUnit]]], 
     Print["Units pb"]]]];

Options[DtKVap] = {DtKVapDefaultUnit -> QuantityUnit[Quantity[("Centimeters")^2/("Seconds")]]};
DtKVap[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &), \[Rho]_?(NumberQ[#] || (QuantityQ[#] && \
(CompatibleUnitQ[#1, ("Grams")/("Centimeters")^3])) &), 
   opts : OptionsPattern[{TtransVapK, NdNtcVapK, DtKVap}]] := 
  With[{kB = 1.38065812*^-16,Temp = If[QuantityQ[T], 
      N@QuantityMagnitude@UnitConvert[T, "Kelvins"], T], 
    Dens = If[QuantityQ[\[Rho]], N@QuantityMagnitude@UnitConvert[\[Rho], ("Grams")/("Centimeters")^3], \[Rho]]}, 
   With[{CoefDiff = Quantity[(Hnd[Temp/TtransVapK[Dens, FilterRules[{opts}, Options[TtransVapK]]],NdNtcVapK[Dens, FilterRules[{opts}, Options[NdNtcVapK]]]]*fqc[Temp, Dens]*qc0K[Dens]*kB*Temp*1.5)/(\[Pi]^2 OptionValue[Coefd]*d Sqrt[OptionValue[CoefK]*K0starlim[Dens]*K0K*Dens]), ("Centimeters")^2/("Seconds")]}, 
    If[CompatibleUnitQ[OptionValue[DtKVapDefaultUnit], ("Centimeters")^2/("Seconds")], 
     If[QuantityQ[T] || QuantityQ[\[Rho]], 
      UnitConvert[CoefDiff, OptionValue[DtKVapDefaultUnit]], 
      QuantityMagnitude@UnitConvert[CoefDiff, OptionValue[DtKVapDefaultUnit]]], 
     Print["Units pb"]]]];

(* ::Input::Initialization:: *)
(*Full mean square displacement<u^2> with units written in u2lDefaultUnit; if units are not precised then \[Rho]=density in g/cm^3 and T=temperature in Kelvin*)
Options[u2lVap] = {u2lVapDefaultUnit -> QuantityUnit[Quantity[("Centimeters")^2]]};
u2lVap[T_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1,"Kelvins"])) &), \[Rho]_?(NumberQ[#] || (QuantityQ[#] && (CompatibleUnitQ[#1, ("Grams")/("Centimeters")^3])) &), 
   opts : OptionsPattern[{TtransVapK, NdNtcVapK, u2lVap}]] := 
  With[{kB = 1.38065812*^-16, Temp = If[QuantityQ[T], N@QuantityMagnitude@UnitConvert[T, "Kelvins"], T], 
    Dens = If[QuantityQ[\[Rho]], 
      N@QuantityMagnitude@UnitConvert[\[Rho], ("Grams")/("Centimeters")^3], \[Rho]]}, 
   With[{MeanSquareDisp = Quantity[(Hnd[Temp/TtransVapK[Dens, FilterRules[{opts}, Options[TtransVapK]]],NdNtcVapK[Dens, FilterRules[{opts}, Options[NdNtcVapK]]]]*fqc[Temp, Dens]*qc0K[Dens]*kB*Temp)/(\[Pi]^2 OptionValue[CoefK]*K0starlim[Dens]*K0K), ("Centimeters")^2]}, 
    If[CompatibleUnitQ[OptionValue[u2lVapDefaultUnit], ("Centimeters")^2], 
     If[QuantityQ[T] || QuantityQ[\[Rho]], 
      UnitConvert[MeanSquareDisp, OptionValue[u2lVapDefaultUnit]], 
      QuantityMagnitude@
       UnitConvert[MeanSquareDisp, OptionValue[u2lVapDefaultUnit]]], 
     Print["Units pb"]]]];

(*******************************)
End[]   (* end private context *)
(*******************************)
Protect["EoSPotassium`*"];

(*******************************)
EndPackage[];
(*******************************)

